<?php

namespace Bagaar\Documentation\Generator;


use Barryvdh\Reflection\DocBlock;
use Illuminate\Database\Eloquent\Model;
use Symfony\Component\Console\Helper\ProgressBar;

class Resources
{
    public function parse($config, ProgressBar $progress)
    {
        return collect($config)
            ->mapWithKeys(function ($definition, $name) use ($progress) {
                $data = [
                    'name'               => $name,
                    'model'              => $definition['model'],
                    'attributes'         => $this->getModelAttributes($definition['model']),
                    'relations'          => $this->normaliseRelations(
                        $definition['model'],
                        $definition['relationships'] ?? null,
                        $definition['visible_relationships'] ?? null
                    ),
                    'validation'         => $this->parseValidationRules($definition['validation_rules'] ?? null),
                    'filters'            => [
                        'type'       => 'and',
                        'attributes' => $definition['filters'] ?? [],
                    ],
                    'include_timestamps' => $definition['timestamps_visible'] ?? false,
                ];

                $data['fillable_fields'] = $this->getFillables($definition['model'], $data['attributes']);

                $progress->advance();

                return [
                    $name => $data,
                ];
            })
            ->toArray();
    }

    protected function getModelAttributes($model)
    {
        try {
            $reflector = new \ReflectionClass($model);
            $docBlock = $reflector->getDocComment();

            $blockParser = new DocBlock($docBlock);

            return collect($blockParser->getTags())
                ->filter(function ($tag) {
                    return get_class($tag) == DocBlock\Tag\PropertyTag::class;
                })
                ->map(function (DocBlock\Tag\PropertyTag $tag) {
                    return [
                        'name'        => $tag->getVariableName(),
                        'type'        => $tag->getTypes(),
                        'description' => $tag->getDescription(),
                    ];
                })
                ->toArray();
        } catch (\ReflectionException $e) {
            return [];
        }
    }

    /**
     * @param string     $modelClass
     * @param array|null $relations
     * @param array|null $relationVisibility
     *
     * @return array
     */
    protected function normaliseRelations($modelClass, $relations, $relationVisibility)
    {
        // Skip if not defined (properly)
        if ($relations == null || !is_array($relations) || count($relations) == 0) {
            return [];
        }

        $visibility = [];

        // Map visibility to relations.
        if (is_array($relationVisibility)) {
            foreach ($relationVisibility as $group => $keys) {
                foreach ($keys as $key) {
                    if (!isset($visibility[$key])) {
                        $visibility[$key] = [];
                    }

                    $visibility[$key][] = $group;
                }
            }
        }

        return collect($relations)
            ->map(function ($model, $method) use ($modelClass, $visibility) {
                $instance = new $modelClass();
                $type = get_class($instance->$method());

                $visible = (count($visibility) == 0 || !isset($visibility[$method])) ?
                    '*' : implode(', ', $visibility[$method]);

                return [
                    'slug'    => $method,
                    'model'   => $model,
                    'type'    => $type,
                    'visible' => $visible,
                ];
            })
            ->toArray();
    }


    /**
     * Retrieve the global validation rules for the defined resource.
     *
     * @param $rules
     *
     * @return array
     */
    protected function parseValidationRules($rules)
    {
        $data = [
            'create' => [],
            'update' => [],
        ];

        // Skip if not defined (properly)
        if ($rules == null || !is_array($rules) || count($rules) == 0) {
            return $data;
        }

        foreach ($rules as $rule => $payload) {
            if (!is_array($payload)) {
                $payload = explode('|', $payload);
            }

            $data['create'][$rule] = $data['update'][$rule] = $payload;
        }

        return $data;
    }

    protected function getFillables($model, $attributes)
    {
        $attributes = collect($attributes)
            ->mapWithKeys(function ($attribute) {
                $name = str_replace('$', '', $attribute['name']);
                $compose = [implode('|', $attribute['type'])];

                return [$name => implode(' ', $compose)];
            })
            ->toArray();

        /**
         * @var $instance Model
         */
        $instance = new $model;

        $properties = [];
        $fillables = $instance->getFillable();
        $excludeProperties = $instance->getGuarded();
        $excludeProperties[] = $instance->getKeyName();

        if (count($fillables) == 1 && $fillables[0] == '*') {
            $properties = $attributes;
        } else {
            foreach ($fillables as $i => $p) {
                $properties[$p] = $attributes[$p];
            }
        }

        if (count($properties) > 0) {
            // Remove guarded properties
            foreach ($excludeProperties as $exclude) {
                if (isset($properties[$exclude])) {
                    unset($properties[$exclude]);
                }
            }
        }

        return $properties;
    }
}