<?php

namespace Bagaar\Documentation\Generator;


use BadMethodCallException;
use Barryvdh\Reflection\DocBlock;
use Symfony\Component\Console\Helper\ProgressBar;

class Controllers
{
    public function parse($controllers, $resources, ProgressBar $progress)
    {
        return collect($controllers)->map(function ($methods) use ($progress, $resources) {
            $model = null;
            $mainController = null;

            $endpoints = collect($methods)
                ->filter(function ($method) {
                    return !str_contains($method['name'], ['relation', 'related']);
                })
                ->map(function ($endPoint) use ($resources, &$model, &$mainController) {
                    $controller = $endPoint['controller']['class'];

                    if ($model == null) {
                        if(!property_exists($controller, 'model')) return;

                        $model = $controller::$model;
                    }
                    if ($mainController == null) {
                        $mainController = $controller;
                    }

                    $transformer = $this->getTransformer($endPoint, $model);

                    $data = [
                        'method'      => $endPoint['method'],
                        'uri'         => $this->getUri($endPoint['uri']),
                        'middleware'  => $endPoint['middleware'],
                        'rules'       => $this->getRules($endPoint, $model, $resources),
                        'filters'     => $endPoint['controller']['method'] == 'index' ?
                            $this->getFilters($endPoint['controller'], $resources, $model) : false,
                        'response'    => $this->getResponse($transformer, $resources[$model]['attributes'],
                            $resources[$model]['include_timestamps']),
                        'attributes'  => $this->getRequestAttributes(
                            $endPoint['method'],
                            $endPoint['controller']['method'],
                            $endPoint['controller']['class'],
                            $resources[$model]['fillable_fields']
                        ),
                        'description' => $this->getDescription(
                            new \ReflectionMethod($endPoint['controller']['class'], $endPoint['controller']['method']),
                            $model
                        ),
                    ];

                    return $data;
                })
                ->filter();
            if($endpoints->count() == 0) return;

            $relations = collect($methods)
                ->filter(function ($method) {
                    return str_contains($method['name'], ['relation', 'related']);
                })
                ->map(function ($relation) {
                    return explode('relationships/', $relation['uri']['path'])[1] ?? null;
                })
                ->filter()
                ->unique()
                ->mapWithKeys(function ($relation) use ($model, $resources) {
                    $rel = $resources[$model]['relations'][$relation] ?? null;

                    if ($rel == null) {
                        return ['n' => null];
                    }

                    return [$rel['slug'] => $rel];
                })
                ->filter()
                ->toArray();

            $description = $this->getDescription(new \ReflectionClass($mainController), $model);
            $progress->advance();

            return compact('endpoints', 'relations', 'model', 'description');
        })
            ->filter()
            ->all();
    }

    /**
     * Get the rules for a specific method.
     *
     * If a custom request class is mentioned or a model-specific request class was created, we'll get the rules from
     * there. Otherwise we'll look at the model's general rules, if applicable.
     *
     * @param array  $data      Endpoint description
     * @param string $model     Model key
     * @param array  $resources Bagaaravel model resources
     *
     * @return array|null|false
     * @throws \ReflectionException
     */
    protected function getRules($data, $model, $resources)
    {
        // Try to add validation rules only to create/update requests
        if (!str_contains($data['method'], ['POST', 'PUT', 'PATCH'])) {
            return false;
        }

        // Get request-specific validation rules
        $rules = $this->normaliseRules($this->parseRules($data, $model));

        // None were found, check the resources' validation rules
        if ($rules == null) {
            // We can only do this for standard create/update actions
            if (!in_array($data['controller']['method'], ['update', 'store'])) {
                return null;
            }

            // Use the model's general rules
            $type = $data['controller']['method'] == 'update' ? 'update' : 'create';

            return $this->normaliseRules($resources[$model]['validation'][$type]);
        }

        return $rules;
    }

    /**
     * Normalises found rules.
     *
     * @param array $fields A map that contains fields <-> validation rules.
     *
     * @return null|array
     */
    protected function normaliseRules($fields)
    {
        if (!is_array($fields) && count($fields) == 0) {
            return null;
        }

        return collect($fields)
            ->map(function ($rules) {
                return collect($rules)
                    ->map(function ($rule) {
                        if(is_object($rule)) $rule = get_class($rule);
                        $ruleExploded = explode(':', $rule);
                        $rule = $ruleExploded[0];
                        $parameters = $ruleExploded[1] ?? null;

                        $response = [
                            'rule' => $rule,
                        ];

                        if ($parameters != null) {
                            $response['parameters'] = $parameters;
                        }

                        return $response;
                    })
                    ->toArray();
            })->toArray();
    }

    /**
     * Retrieves the request class bound to a method and returns its rules.
     *
     * @param array  $action Contains the controller's class and method.
     * @param string $model  Model resource key
     *
     * @return null|array
     * @throws \ReflectionException
     */
    protected function parseRules($action, $model)
    {
        // Get the method's parameters
        $parameters = collect(
            (new \ReflectionMethod($action['controller']['class'], $action['controller']['method']))
                ->getParameters()
        );

        // Get the request parameter
        $request = $parameters->filter(function (\ReflectionParameter $parameter) {
            return $parameter->getName() == 'request';
        })
            ->first();
        if ($request == null) {
            return null;
        }

        // Retrieve the class that's bound to the request variable
        $requestClassName = $request->getType()->getName();

        if ($requestClassName == null) {
            return null;
        }

        // Resolve the request class name to an implementation
        $requestClass = ($requestClassName == 'Bagaaravel\Api\Requests\FormRequestInterface') ?
            $this->resolveRequest($action, $model) : $requestClassName;

        if ($requestClass == null) {
            return null;
        }

        // Try to get the validation rules from the request
        try {
            return (new $requestClass)->rules();
        }
            // Method didn't exist
        catch(BadMethodCallException $e)
        {
            return null;
        }
    }

    /**
     * Tries to look for a model/action specific request class.
     *
     * @param array  $action Contains the controller's class and method.
     * @param string $model  Model resource key
     *
     * @return null|string
     */
    protected function resolveRequest($action, $model)
    {
        $method = $action['controller']['method'];
        $formRequest = ucfirst($model) . ucfirst($method) . 'Request';

        if (class_exists($this->getAppNamespace() . "Http\\Requests\\${formRequest}")) {
            return $this->getAppNamespace() . "Http\\Requests\\${formRequest}";
        }

        return null;
    }

    /**
     * Parses the uri to a unified format.
     *
     * @param array $definition The endpoint's uri property.
     *
     * @return string
     */
    protected function getUri($definition)
    {
        $uri = $definition['path'];

        // Replace the identifier with a generic :id
        if ($definition['identifier'] != null) {
            return str_replace($definition['identifier'], ':id', $uri);
        }

        return $uri;
    }

    protected function getAppNamespace()
    {
        return app()->getNamespace();
    }

    /**
     * Retrieves the method's transformer class.
     *
     * @param array  $data  The method's definition
     * @param string $model The model resource key
     *
     * @return null|string
     */
    protected function getTransformer($data, $model)
    {
        if (str_contains($data['method'], 'DELETE')) {
            return null;
        }

        try {
            $reflector = new \ReflectionMethod($data['controller']['class'], $data['controller']['method']);
            $docBlock = $reflector->getDocComment();

            $blockParser = new DocBlock($docBlock);

            $tags = $blockParser->getTags();

            if (count($tags) == 0) {
                return ucfirst($model) . 'Transformer';
            }

            return collect($tags)
                ->filter(function ($tag) {
                    return get_class($tag) == DocBlock\Tag::class && $tag->getName() == 'transformer';
                })
                ->map(function ($tag) {
                    return $tag->getDescription();
                })
                ->first() ?: ucfirst($model) . 'Transformer';
        } catch (\ReflectionException $e) {
            return ucfirst($model) . 'Transformer';
        }
    }

    /**
     * Get a class/method's description.
     *
     * @param        $reflector The reflector used for retrieval
     * @param string $model     The model resource key
     *
     * @return string|null
     */
    protected function getDescription($reflector, $model)
    {
        $docBlock = $reflector->getDocComment();

        $blockParser = new DocBlock($docBlock);

        if ($blockParser == false) {
            return null;
        }

        $description = $blockParser->getShortDescription();
        $longDescription = $blockParser->getLongDescription();
        if (!empty($longDescription)) {
            $description .= "\n" . $longDescription;
        }

        return str_replace(['resource', 'Resource'], $model, $description);

    }

    /**
     * @param        $data
     * @param        $resources
     * @param string $model The model resource key
     *
     * @return null
     */
    protected function getFilters($data, $resources, $model)
    {
        return $resources[$model]['filters']['attributes'];
    }

    /**
     * Try to get an endpoints response payload.
     *
     * @param string  $transformer       The endpoint's transformer
     * @param array   $modelAttributes   Fallback attributes
     * @param boolean $includeTimeStamps Should timestamps be included in the payload?
     *
     * @return null|array
     */
    protected function getResponse($transformer, $modelAttributes, $includeTimeStamps)
    {
        // No transformer was defined, so we're expecting no result.
        if ($transformer == null || $transformer == 'null') {
            return null;
        }

        // Normalise model attributes
        $modelAttributes = collect($modelAttributes)
            ->map(function ($attribute) {
                $attribute['name'] = str_replace('$', '', $attribute['name']);
                $attribute['type'] = implode('|', $attribute['type']);
                $attribute['description'] = $attribute['description'] ?: null;

                return $attribute;
            })
            ->filter(function ($attribute) use ($includeTimeStamps) {
                $exclude = ['id'];

                if (!$includeTimeStamps) {
                    $exclude = array_merge($exclude, ['created_at', 'updated_at', 'deleted_at']);
                }

                return !in_array($attribute['name'], $exclude);
            })
            ->toArray();

        // Fallback response
        $response = [
            'data' => [
                'id'         => 'integer',
                'attributes' => $this->placePayloadDataMultilevel($modelAttributes),
            ],
        ];

        // Check if the transformer class name can be found
        if (!class_exists($transformer)) {
            // Might not have gotten the right prefix
            $transformer = 'App\Transformers\\' . $transformer;
        }

        // Still doesn't exist
        if (!class_exists($transformer)) {
            // Return the model's attributes
            return $response;
        }


        try {
            $reflector = new \ReflectionMethod($transformer, 'transform');
            $attributes = $this->getReflectionProperties($reflector);

            if ($attributes == false) {
                return $response;
            }

            // Parse the transformer's properties
            $response['data']['attributes'] = $this->placePayloadDataMultilevel($attributes);

            return $response;
        } catch (\ReflectionException $e) {
            // Return the fallback response
            return $response;
        }
    }

    /**
     * Add support for dot syntax.
     *
     * @param array $payload Payload to parse.
     *
     * @return array
     */
    protected function placePayloadDataMultilevel($payload)
    {
        $attributes = [];

        foreach ($payload as $description) {
            array_set($attributes, $description['name'], array_only($description, ['type', 'description']));
        }

        return $attributes;
    }

    /**
     * @param string $requestMethod
     * @param string $classMethod
     * @param string $controller
     * @param array  $fillableFields
     *
     * @return array|null
     */
    protected function getRequestAttributes($requestMethod, $classMethod, $controller, $fillableFields)
    {
        if (in_array($classMethod, ['store', 'update'])) {
            return [
                'data' => [
                    'attributes' => $fillableFields,
                ],
            ];
        }

        if (str_contains($requestMethod, ['POST', 'PATCH', 'PUT'])) {
            try {
                $reflector = new \ReflectionMethod($controller, $classMethod);

                return [
                    'data' => [
                        'attributes' => collect($this->getReflectionProperties($reflector))
                            ->mapWithKeys(function ($property) {
                                return [
                                    $property['name'] => $property['type'],
                                ];
                            })
                            ->toArray(),
                    ],
                ];
            } catch (\ReflectionException $e) {
                return null;
            }
        }

        return null;
    }

    /**
     * Get a class or method's '@property' definitions.
     *
     * @param \ReflectionFunctionAbstract $reflector
     *
     * @return bool|array
     */
    protected function getReflectionProperties(\ReflectionFunctionAbstract $reflector)
    {
        $docBlock = $reflector->getDocComment();

        $blockParser = new DocBlock($docBlock);

        if ($blockParser == false) {
            return false;
        }

        // Parse the transformer's properties
        return collect($blockParser->getTags())
            ->filter(function ($tag) {
                return get_class($tag) == DocBlock\Tag\PropertyTag::class;
            })
            ->map(function (DocBlock\Tag\PropertyTag $tag) {
                return [
                    'name'        => str_replace('$', '', $tag->getVariableName()),
                    'type'        => implode('|', $tag->getTypes()),
                    'description' => $tag->getDescription() ?: null,
                ];
            })
            ->toArray();

    }
}