<?php

namespace Bagaar\Documentation\Commands;

use Bagaar\Documentation\Facades\Documentation;
use Illuminate\Console\Command;

class InstallDocumentationGenerator extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'documentation:install';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Prepares your local app to generate API documentation.';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $this->warn('[!] Installing VuePress');
        $this->installVuePress();
        $this->warn('[!] Creating documentation structure');
        $this->addSkeleton();
    }

    protected function installVuePress()
    {
        // Prepare NPM/Yarn
        $packagesFile = base_path('package.json');

        $this->info('[+] Preparing package.json');
        if (!file_exists($packagesFile)) {
            $packages = [
                'scripts'         => [],
                'devDependencies' => [],
            ];
        } else {
            $packages = json_decode(file_get_contents($packagesFile), true);
        }

        $packages['devDependencies']['vuepress'] = '^0.13.1';
        $packages['devDependencies']['lodash'] = '^3.7.0';

        $packages['scripts']['docs:dev'] = 'vuepress dev resources/docs';
        $packages['scripts']['docs:build'] = 'vuepress build resources/docs';

        // Write new packages.json
        file_put_contents($packagesFile, json_encode($packages, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        $this->info('[+] Set up resources');

        // Create directories
        $directories = [
            resource_path('docs'),
            resource_path('docs/.vuepress'),
            resource_path('docs/.vuepress/components'),
        ];

        foreach ($directories as $dir) {
            if (!file_exists($dir)) {
                mkdir($dir);
            }
        }

        // Add gitignore
        file_put_contents(
            $directories[0] . '/.gitignore',
            '*.md
                  .vuepress/*js*
                  .vuepress/components'
        );
    }

    protected function addSkeleton()
    {
        $docsDir = resource_path('docs');

        $this->info('[+] Creating docs config file');
        Documentation::installer('config', $docsDir)->store($this);

        $this->info('[+] Add front-page');
        Documentation::installer('home', $docsDir)->store($this);

        $this->info('[+] Add components');
        Documentation::installer('components', $docsDir)->store($this);
    }
}
