<?php

namespace Bagaar\Documentation\Commands;

use Bagaar\Documentation\Facades\Documentation;
use Illuminate\Console\Command;
use Illuminate\Routing\Router;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Input\ArrayInput;

class GenerateDocumentation extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'gen:documentation {--M|models : Add this if you want to regenerate your model properties before generating the documentation}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generates the resource files to build the documentation.';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @throws \Throwable
     * @return mixed
     */
    public function handle()
    {
        // Support enum fields
        \DB::getDoctrineSchemaManager()->getDatabasePlatform()->registerDoctrineTypeMapping('enum', 'string');

        if($this->option('models'))
        {
            $this->info('+ Generate model properties');

            // Attach a model's properties as doc-block for easy reference
            $command = $this->getApplication()->find('ide-helper:models');
            $input = new ArrayInput([
                '-W' => true
            ]);
            $command->run($input, $this->output);
        }

        $this->info('+ Normalise Bagaaravel models (resources)');
        $resources = config('bagaaravel.jsonapi');
        $progress = new ProgressBar($this->output, count($resources));

        // Normalise the model resources from the config file
        $resources = Documentation::generator('resources')->parse($resources, $progress);
        $progress->finish();
        $this->info('');

        // Normalise routes (retrieves API and oAuth routes)
        $this->info('+ Read out routes');
        $router = app(Router::class)->getRoutes();
        $progress = new ProgressBar($this->output, count($router));

        $routes = Documentation::generator('routes')->parse($router, $progress);
        $progress->finish();
        $this->info('');

        // Connect controllers to their respective routes and resources
        $this->info('+ Connect controllers to resources');
        $progress = new ProgressBar($this->output, count($routes));

        $endpoints = Documentation::generator('controllers')->parse($routes, $resources, $progress);
        $progress->finish();
        $this->info('');


        // Render the resources
        $this->info('+ Generate resource file');
        file_put_contents(
            resource_path('docs/resources.md'),
            view('documentation::resources')
                ->with(compact('resources'))
                ->render()
        );

        // Render the resources
        $this->info('+ Generate endpoints file');
        file_put_contents(
            resource_path('docs/endpoints.md'),
            view('documentation::endpoints')
                ->with(compact('endpoints'))
                ->render()
        );
    }
}
