<?php

namespace Bagaaravel\Api;

use Bagaaravel\Exceptions\ApiException;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Input;
use LucaDegasperi\OAuth2Server\Facades\Authorizer;

trait CRUDEndpoint
{
    use ApiGateway;

    /**
     * @var mixed
     * Eloquent model we're working with, set on construct
     */
    private $resource;
    /**
     * @var array of validation rules
     * see: https://laravel.com/docs/5.2/validation#available-validation-rules
     */
    private $validation_rules = [];
    /**
     * @var mixed
     * Collection we're working with, you can override this in controller
     * when scope is more specific than ->all()
     */
    private $collection = false;
    /**
     * @var array
     * Attributes from the request
     */
    private $attributes = [];

    /**
     * @var string
     * Attribute on the model that defines parent
     */
    private $parent_key = null;
    /**
     * @var array
     * Attribute on the model that is linked to current user session id
     */
    private $user_mapping = null;

    /**
     * Display a listing of the resource
     *
     * @return Response
     */
    public function index($parent_id = null)
    {
        $items = [];
        if (!$this->collection) {
            $this->collection = $this->resource;
            if ($parent_id) {
                // scope to parent_key if set
                $this->collection = $this->collection->where($this->parent_key, $parent_id);
            }
        }
        foreach ($this->collection->get() as $item) {
            $items[] = $this->transform($item);
        }

        return $this->respondCollection($items);
    }

    /**
     * Store a new resource
     *
     * @return Response
     */
    public function store($parent_id = null)
    {
        $this->attributes = Input::all();
        if ($this->parent_key) {
            $this->attributes[$this->parent_key] = $parent_id;
        }
        $this->validateInput();

        $item = $this->resource->create($this->attributes);
        if ($this->user_mapping) {
            $item[$this->user_mapping] = Authorizer::getResourceOwnerId();
            $item->save();
        }
        $this->handleRelations($item);

        return $this->respondItem($this->transform($item->fresh()), Response::HTTP_CREATED);
    }

    /**
     * Display the specified resource
     *
     * @param  int  $id
     * @return Response
     */
    public function show(...$ids)
    {
        $item = $this->findResourceAndAuthorize($ids);

        return $this->respondItem($this->transform($item));
    }

    /**
     * Update the specified resource
     *
     * @param  int  $id
     * @return Response
     */
    public function update(...$ids)
    {
        $this->attributes = Input::all();
        if ($this->parent_key) {
            $this->attributes[$this->parent_key] = $ids[0];
        }
        $this->validateInput();
        $item = $this->findResourceAndAuthorize($ids);

        $item->update($this->attributes);
        $this->handleRelations($item);

        return $this->respondItem($this->transform($item->fresh()));
    }

    /**
     * Remove the specified resource
     *
     * @param  int  $id
     * @return Response
     */
    public function destroy(...$ids)
    {
        $item = $this->findResourceAndAuthorize($ids);

        $item->delete();
    }

    /**
     * Find and authorize the resource
     *
     * @param  int  $id
     * @return Resource
     */
    private function findResourceAndAuthorize($ids)
    {
        # @TODO scope this correctly
        $id   = $ids[count($ids) - 1];
        $item = $this->resource->find($id);

        if (!$item) {
            throw new ApiException('NOT_FOUND', Response::HTTP_NOT_FOUND);
        }

        if (!$this->authorize($item)) {
            throw new ApiException('UNAUTHORIZED', Response::HTTP_UNAUTHORIZED);
        }

        return $item;
    }

    /**
     * Authorization ruling
     *
     * @param  Resource  $resource
     * @return boolean
     */
    private function authorize($resource)
    {
        return true;
    }

    /**
     * Override method to handle relations
     *
     * @param  Item $item
     * @return Item
     */
    private function handleRelations($item)
    {
        return $item;
    }

}
