<?php namespace Bagaaravel\Api;

use Bagaaravel\Exceptions\JsonApiErrorException;
use Config;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Validator;

trait JsonApiTrait
{
    protected $page_size     = 10;
    protected $visible_atts  = [];
    protected $hidden_atts   = [];
    protected $atts          = [];
    protected $validation    = [];
    protected $filters       = [];
    protected $relationships = false;
    protected $record        = null;
    protected $result_set    = null;
    protected $model         = null;
    protected $route         = null;
    protected $response      = null;
    protected $includes      = [];

    public function __construct()
    {
        $cfg            = Config::get('bagaaravel.jsonapi.' . $this->model);
        $this->resource = new $cfg['model'];
        if (isset($cfg['relationships'])) {
            $this->relationships = $cfg['relationships'];
        }
        if (isset($cfg['filters'])) {
            $this->filters = $cfg['filters'];
        }
        if (isset($cfg['hidden'])) {
            $this->hidden_atts = $cfg['hidden'];
        }
        if (isset($cfg['visible'])) {
            $this->visible_atts = $cfg['visible'];
        }
    }

    public function setupScope()
    {
        $this->collection = !isset($this->collection) ? $this->resource : $this->collection;
    }

    public function transformItem($item)
    {
        // see my comment here: https://github.com/laravel/internals/issues/86
        $item->addHidden($this->relationships);
        $item->addHidden(array_merge($this->hidden_atts, ['id']));

        $output = [
            'type'       => $this->model,
            'id'         => $item['id'],
            'links'      => [
                'self' => route($this->route . '.show', $item['id']),
            ],
            'attributes' => $item
                ->makeVisible($this->visible_atts)
                ->toArray(),
        ];

        if ($this->relationships) {
            // i know, not the cleanest, work in progress, ready for refactor
            $relations = [];
            foreach ($this->relationships as $relation) {
                $include = in_array($relation, $this->includes);
                $value   = $item->{$relation};
                if (!$value) {
                    continue;
                }

                if ($value instanceof Collection) {
                    $items = ['data' => []];
                    foreach ($value as $item) {
                        $data = [
                            'id'   => (string) $item->getKey(),
                            'type' => $item->getTable(),
                        ];
                        $items['data'][] = $data;
                        if ($include) {
                            $data['attributes'] = $item->toArray();
                            $this->response->addIncludes($data);
                        }
                    }
                    $relations[$relation] = $items;
                } else {
                    $data = [
                        'id'   => (string) $value->getKey(),
                        'type' => $value->getTable(),
                    ];
                    $relations[$relation] = ['data' => $data];
                    if ($include) {
                        $data['attributes'] = $value->toArray();
                        $this->response->addIncludes($data);
                    }
                }
            }
            $output['relationships'] = $relations;
        }

        return $output;
    }

    public function validateInput()
    {
        $input = Input::only('data.attributes');
        if (!isset($input['data']['attributes'])) {
            throw new JsonApiErrorException('Invalid request', Response::HTTP_UNPROCESSABLE_ENTITY);
        }
        $this->atts = $input['data']['attributes'];
        $this->type = Input::only('data.type');
        $validator  = Validator::make($this->atts, $this->validation);
        if ($validator->fails()) {
            throw new JsonApiErrorException($validator->messages(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function handleRelations()
    {
        if (!$this->relationships) {
            // no relationships are allowed, so skip handling them
            return;
        }

        $input = Input::only('data.relationships');
        if (!isset($input['data']['relationships'])) {
            return;
        }
        $relations = $input['data']['relationships'];

        foreach ($relations as $relation) {
            $item = $relation['data'];
            if (!in_array($item['type'], $this->relationships)) {
                continue;
            }
            $key = $this->relationships[$item['type']];
            $this->record->update([$key => $item['id']]);
        }
    }

    public function handleFilters()
    {
        foreach (Input::get('filter', []) as $key => $value) {
            if (in_array($key, $this->filters)) {
                if (strrpos($value, '%') === false) {
                    $this->collection = $this->collection->where($key, '=', $value);
                } else {
                    $this->collection = $this->collection->where($key, 'LIKE', $value);
                }
            }
        }
    }

    public function findResourceAndAuthorize($id)
    {
        $this->record = $this->resource->find($id);

        if (!$this->record) {
            throw new JsonApiErrorException('NOT_FOUND', Response::HTTP_NOT_FOUND);
        }

        if (!$this->authorize()) {
            throw new JsonApiErrorException('UNAUTHORIZED', Response::HTTP_UNAUTHORIZED);
        }
    }

    public function authorize()
    {
        return true;
    }

    public function buildResponse($action)
    {
        $this->response = new JsonApiResponse;
        $res            = $this->result_set ? $this->result_set : $this->record;

        if ($this->relationships) {
            $includes = Input::get('include', false);
            if ($includes) {
                $includes = explode(',', $includes);
                foreach ($includes as $item) {
                    if (!in_array($item, $this->relationships)) {
                        continue;
                    }
                    $this->includes[] = $item;
                }
            }
        }

        $data = [];

        if (is_a($res, 'Illuminate\Pagination\LengthAwarePaginator')) {

            // handle collection
            $this->response->addLinks([
                'self'  => $res->url($res->currentPage()),
                'first' => $res->url(1),
                'prev'  => $res->previousPageUrl(),
                'next'  => $res->nextPageUrl(),
                'last'  => $res->url($res->lastPage()),
            ]);

            $this->response->addMeta([
                'total_records' => $res->total(),
                'total_pages'   => $res->lastPage(),
                'page_size'     => $this->page_size,
            ]);

            foreach ($res as $item) {
                $data[] = $this->transformItem($item);
            }

            $this->response->setData($data);

        } else {

            // handle single item
            $this->response->setData($this->transformItem($this->record));

            $this->response->addLinks([
                'self' => route($this->route . '.show', $res->id),
            ]);

        }

        switch ($action) {
            case 'store':
                $res = $this->response->output();
                return $this->respond($res, Response::HTTP_CREATED, [
                    'Location' => $res['links']['self'],
                ]);
            case 'destroy':
                return $this->respond(null, Response::HTTP_NO_CONTENT);
            default:
                return $this->respond($this->response->output());
                break;
        }
    }

    public function respond($object, $status = 200, $headers = [])
    {
        if ($object) {
            $res = response()->json($object, $status)->withHeaders(
                array_merge($headers, ['Content-Type' => 'application/vnd.api+json'])
            );
        } else {
            $res = response(null, $status);
        }

        return $res;
    }

}
