<?php namespace Bagaaravel\Api;

use Authorizer;
use Bagaaravel\Exceptions\JsonApiErrorException;
use Config;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Validator;

trait JsonApiTrait
{
    protected $page_size     = 10;
    protected $visible_atts  = [];
    protected $hidden_atts   = [];
    protected $atts          = [];
    protected $validation    = [];
    protected $filters       = [];
    protected $relationships = false;
    protected $check_auth    = true;
    protected $record        = null;
    protected $result_set    = null;
    protected $model         = null;
    protected $route         = null;
    protected $response      = null;
    protected $owner         = null;
    protected $includes      = [];

    public function __construct()
    {
        $this->cfg      = Config::get('bagaaravel.jsonapi');
        $model_config   = $this->cfg[$this->model];
        $this->resource = new $model_config['model'];
        if (isset($model_config['relationships'])) {
            $this->relationships = $model_config['relationships'];
        }
        if (isset($model_config['filters'])) {
            $this->filters = $model_config['filters'];
        }
        if (isset($model_config['hidden'])) {
            $this->hidden_atts = $model_config['hidden'];
        }
        if (isset($model_config['visible'])) {
            $this->visible_atts = $model_config['visible'];
        }

        $this->owner = Authorizer::getResourceOwnerId();
    }

    public function transformItem($item)
    {
        // see my comment here: https://github.com/laravel/internals/issues/86
        $item->addHidden(
            array_merge($this->hidden_atts, array_keys($this->relationships), ['id'])
        );

        $output = [
            'type'       => $this->model,
            'id'         => $item['id'],
            'links'      => [
                'self' => route($this->route . '.show', $item['id']),
            ],
            'attributes' => $item
                ->makeVisible($this->visible_atts)
                ->toArray(),
        ];

        if ($this->relationships) {
            // i know, not the cleanest, work in progress, ready for refactor
            $relations = [];
            foreach ($this->relationships as $relation => $type) {
                $include = in_array($relation, $this->includes);
                $value   = $item->{$relation};
                if (!$value) {
                    continue;
                }

                if ($value instanceof Collection) {
                    $items = ['data' => []];
                    foreach ($value as $item) {
                        $data = [
                            'id'   => (string) $item->getKey(),
                            'type' => $type,
                        ];
                        $items['data'][] = $data;
                        if ($include) {
                            $data['attributes'] = $item->toArray();
                            $this->response->addIncludes($data);
                        }
                    }
                    $relations[$relation] = $items;
                } else {
                    $data = [
                        'id'   => (string) $value->getKey(),
                        'type' => $type,
                    ];
                    $relations[$relation] = ['data' => $data];
                    if ($include) {
                        $data['attributes'] = $value->toArray();
                        $this->response->addIncludes($data);
                    }
                }
            }
            $output['relationships'] = $relations;
        }

        return $output;
    }

    public function validateInput()
    {
        $input = Input::only('data.attributes');
        if (!isset($input['data']['attributes'])) {
            throw new JsonApiErrorException('Invalid request', Response::HTTP_UNPROCESSABLE_ENTITY);
        }
        $this->atts = $input['data']['attributes'];
        $this->type = Input::only('data.type');
        $validator  = Validator::make($this->atts, $this->validation);
        if ($validator->fails()) {
            throw new JsonApiErrorException($validator->messages(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function handleRelations()
    {
        if (!$this->relationships) {
            // no relationships are allowed, so skip handling them
            return;
        }

        $input = Input::only('data.relationships');
        if (!isset($input['data']['relationships'])) {
            // data is not as we need it
            return;
        }

        $relations = $input['data']['relationships'];
        foreach ($relations as $key => $data) {
            if (!in_array($key, array_keys($this->relationships))) {
                continue;
            }
            $this->updateRelations($key, $data);
        }

    }

    public function updateRelations($key, $data)
    {
        // make sure all inverse relations are set
        $inverse      = $this->cfg[$this->relationships[$key]];
        $inverse_key  = array_search($this->model, $inverse['relationships']);
        $owner        = isset($inverse['owner']) ? $inverse['owner'] : 'user_id';
        $RelatedModel = new $inverse['model'];
        $relation     = $this->record->{$key}();
        $class_exp    = explode('\\', get_class($relation));
        $kind         = array_pop($class_exp);

        // single or multiple items
        $ids_to_relate = [];
        if (isset($data['data']['type'])) {
            $ids_to_relate[] = $data['data']['id'];
        } else if ($data['data']) {
            foreach ($data['data'] as $item) {
                $ids_to_relate[] = $item['id'];
            }
        }

        // todo disable scope for admin
        $items_to_relate = $RelatedModel::whereIn('id', $ids_to_relate);
        if ($this->check_auth) {
            $items_to_relate = $items_to_relate->where([$owner => $this->owner]);
        }
        $items_to_relate = $items_to_relate->get();

        switch ($kind) {
            case 'HasMany':
            case 'HasManyThrough':
                foreach ($relation->get() as $child) {
                    if (!in_array($child->id, $ids_to_relate)) {
                        $child->{$inverse_key}()->dissociate();
                        $child->save();
                    }
                }
                foreach ($items_to_relate as $item) {
                    $item->{$inverse_key}()->associate($this->record);
                    $item->save();
                }
                break;
            case 'HasOne':
                if (!in_array($relation->id, $ids_to_relate)) {
                    $relation->{$inverse_key}()->dissociate();
                }
                foreach ($items_to_relate as $item) {
                    $relation->{$inverse_key}()->associate($this->record);
                    $relation->save();
                }
                $relation->save();
                break;
            case 'BelongsTo':
                if (!in_array($this->record->{$key}->id, $ids_to_relate)) {
                    $relation->dissociate();
                    $this->record->save();
                }
                foreach ($items_to_relate as $item) {
                    $relation->associate($item);
                    $this->record->save();
                }
                break;
        }

    }

    public function handleFilters()
    {
        foreach (Input::get('filter', []) as $key => $value) {
            if (in_array($key, $this->filters)) {
                if (strrpos($value, '%') === false) {
                    $this->collection = $this->collection->where($key, '=', $value);
                } else {
                    $this->collection = $this->collection->where($key, 'LIKE', $value);
                }
            }
        }
    }

    public function findResourceAndAuthorize($id)
    {
        $this->record = $this->resource->find($id);

        if (!$this->record) {
            throw new JsonApiErrorException('NOT_FOUND', Response::HTTP_NOT_FOUND);
        }

        if (!$this->authorize()) {
            throw new JsonApiErrorException('UNAUTHORIZED', Response::HTTP_UNAUTHORIZED);
        }
    }

    public function authorize()
    {
        // default authorization is true, override in controller
        return true;
    }

    public function overrides()
    {
        // array with attributes that are always overridden,
        // no matter the user input for these.
        // f.e. user mapping attributes.
        return [];
    }

    public function buildResponse($action)
    {
        $this->response = new JsonApiResponse;
        $res            = $this->result_set ? $this->result_set : $this->record;

        if ($this->relationships) {
            $includes = Input::get('include', false);
            if ($includes) {
                $includes = explode(',', $includes);
                foreach ($includes as $item) {
                    if (!in_array($item, array_values($this->relationships))) {
                        continue;
                    }
                    $this->includes[] = $item;
                }
            }
        }

        $data = [];

        if (is_a($res, 'Illuminate\Pagination\LengthAwarePaginator')) {

            // handle collection
            $this->response->addLinks([
                'self'  => $res->url($res->currentPage()),
                'first' => $res->url(1),
                'prev'  => $res->previousPageUrl(),
                'next'  => $res->nextPageUrl(),
                'last'  => $res->url($res->lastPage()),
            ]);

            $this->response->addMeta([
                'total_records' => $res->total(),
                'total_pages'   => $res->lastPage(),
                'page_size'     => $this->page_size,
            ]);

            foreach ($res as $item) {
                $data[] = $this->transformItem($item);
            }

            $this->response->setData($data);

        } else {

            // handle single item
            $this->response->setData($this->transformItem($this->record));

            $this->response->addLinks([
                'self' => route($this->route . '.show', $res->id),
            ]);

        }

        switch ($action) {
            case 'store':
                $res = $this->response->output();
                return $this->respond($res, Response::HTTP_CREATED, [
                    'Location' => $res['links']['self'],
                ]);
            case 'destroy':
                return $this->respond(null, Response::HTTP_NO_CONTENT);
            default:
                return $this->respond($this->response->output());
                break;
        }
    }

    public function respond($object, $status = 200, $headers = [])
    {
        if ($object) {
            $res = response()->json($object, $status)->withHeaders(
                array_merge($headers, ['Content-Type' => 'application/vnd.api+json'])
            );
        } else {
            $res = response(null, $status);
        }

        return $res;
    }

}
