<?php

namespace Bagaaravel\Api\Gateways;

use Bagaaravel\Api\Repositories\GenericRepository;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * Class GenericGateway
 * @package Bagaaravel\Api\Gateways
 */
class GenericGateway implements GatewayInterface
{
    public $repository;
    protected $model;
    protected $config;

    protected $relationshipsToSave = [];

    public function __construct(GenericRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * @param $model
     */
    public function setConfig($model)
    {
        $this->model = $model;
        $this->config = config("bagaaravel.jsonapi.${model}");
        $this->repository->setModel($this->config['model']);
    }

    public function setFilters($filters)
    {
        return $this->repository->setFilters($filters);
    }

    public function setSorting($sorting)
    {
        return $this->repository->setSorting($sorting);
    }

    /**
     * @return mixed
     */
    public function getAll()
    {
        return $this->repository->getAll();
    }

    /**
     * @param $sorting
     * @return mixed
     */
    public function getAllSorted($sorting)
    {
        return $this->repository->getAllSorted($sorting);
    }

    /**
     * @param int $per_page
     * @return mixed
     */
    public function getAllPaginated($per_page = 20 )
    {
        return $this->repository->getAllPaginated($per_page);
    }

    /**
     * @param $scope
     * @return mixed
     */
    public function getAllWhere($scope)
    {
        return $this->repository->getAllWhere($scope);
    }

    /**
     * @param $scope
     * @param int $per_page
     */
    public function getAllWherePaginated($scope, $per_page = 20)
    {
        return $this->repository->getAllWherePaginated($scope, $per_page);
    }

    /**
     * @param $id
     * @return mixed
     */
    public function getById($id)
    {
        return $this->repository->getById($id);
    }

    /**
     * @param $relationship_type
     * @param $relationship_name
     * @param $relationship_config_name
     * @param $relationship_id
     */
    public function addRelationshipToSave($relationship_type, $relationship_name, $relationship_config_name, $relationship_id, $relationship_meta)
    {
        $this->relationshipsToSave[] = [
            'type' => $relationship_type,
            'name' => $relationship_name,
            'config' => $relationship_config_name,
            'id' => $relationship_id,
            'meta' => $relationship_meta
        ];
    }


    /**
     * @param $model
     * @param $input
     * @return mixed
     */
    protected function saveRelationshipsThatFillAForeignKey($model, $input)
    {
        foreach( $this->relationshipsToSave as $relationship ){
            switch ( $relationship['type'] ){
                case 'BelongsTo':
                    $foreignKey = $model->{$relationship['name']}()->getForeignKey();
                    $model->forceFill([$foreignKey => $relationship['id']]);
                    break;
            }
        }

        return $model;
    }


    /**
     * @param $model
     * @return mixed
     */
    protected function saveRelationshipsThatAreChildren($model)
    {
        foreach( $this->relationshipsToSave as $relationship ){

            $child = $model->{$relationship['name']}()->getModel()->find($relationship['id']);
            $relationship_config = config('bagaaravel.jsonapi.' . $relationship['config']);
            $child_relation_method = array_search(class_basename($model), $relationship_config['relationships']);

            switch ( $relationship['type'] ){

                case 'HasMany':
                case 'HasOne':
                    $child->{$child_relation_method}()->associate($model);
                    $child->save();
                    break;
                case 'BelongsToMany':
                case 'MorphToMany':
                    $child->{$child_relation_method}()->attach($model, $relationship['meta']);
                    break;
            }
        }

        return $model;
    }


    /**
     * @param $input
     * @return mixed
     */
    protected function saveWithRelationships($input, $update_id = null)
    {
        if ( $update_id ){
            $model = $this->repository->getById($update_id);
        } else {
            $model = $this->repository->create();
        }

        //check if own model is polymorphic
        $relationship_config = config('bagaaravel.jsonapi.' . $this->relationshipsToSave[0]['config']);
        $relationship_model_method_name = array_search($this->model, $relationship_config['relationships']);
        $relationship_model_instance = app($relationship_config['model']);
        $relationship_type = class_basename($relationship_model_instance->{$relationship_model_method_name}());

        if ( $relationship_type == 'MorphMany' ){
            $model->fill($input);

            $relationship_model = $relationship_model_instance->find($this->relationshipsToSave[0]['id']);
            $relationship_model->{$relationship_model_method_name}()->save($model);
        } else {
            $model = $this->saveRelationshipsThatFillAForeignKey($model, $input);
            if ( $update_id && empty($input) ){
                //we are just updating relationships
            } else {
                $model->fill($input);
            }
            $model->save();

            $model = $this->saveRelationshipsThatAreChildren($model);

            $this->relationshipsToSave = [];
        }

        return $model;
    }

    /**
     * @param $input
     * @return bool|mixed
     */
    public function save($input)
    {
        //todo allow to set custom methodName
        //todo refactor to relationshipsToSave
        $methodName = str_plural(camel_case($this->model));
        if (method_exists(auth()->user(), $methodName) && auth()->user()->{$methodName}() instanceof HasMany) {
            return auth()->user()->{$methodName}()->create($input);
        }

        if ( count( $this->relationshipsToSave ) ){
            return $this->saveWithRelationships($input);
        }
        return $this->repository->create($input);
    }

    /**
     * @param $id
     * @param $input
     * @return bool|mixed
     */
    public function update($id, $input)
    {
        if ( count( $this->relationshipsToSave ) ){
            return $this->saveWithRelationships($input, $id);
        }

        return $this->repository->save($id, $input);
    }

    /**
     * @param $id
     * @return mixed
     */
    public function delete($id)
    {
        return $this->repository->delete($id);
    }
}