<?php
namespace Bagaaravel\Api;

use Bagaaravel\Api\Gateways\GenericGateway;
use Bagaaravel\Api\Requests\FormRequestInterface;
use Bagaaravel\Api\Responders\JsonApiResponder;
use Bagaaravel\Api\Responders\ResponderInterface;
use Illuminate\Support\Collection;

abstract class JsonApiController extends JsonCrudController
{

    /**
     * JsonApiController constructor.
     * @param GenericGateway $gateway
     * @param ResponderInterface $responder
     */
    function __construct(GenericGateway $gateway, ResponderInterface $responder)
    {
        parent::__construct($gateway, $responder);
    }


    /**
     * @param $kind_of_request
     * @param $relationship_type
     * @param $remote_relationship_type
     */
    protected function checkKindOfRequest($kind_of_request, $relationship_type, $remote_relationship_type)
    {
        switch ( $kind_of_request ){
            case 'updateByResource':

                switch ($relationship_type){
                    case 'HasMany':
                    case 'BelongsToMany':
                    case 'MorphMany':
                    case 'MorphToMany':
                        app()->abort(403);
                        break;
                    case 'BelongsTo':
                        if ( $remote_relationship_type == 'HasOne' ) app()->abort(403);
                        break;

                    case 'HasOne':
                        if ( $remote_relationship_type == 'BelongsTo' ) app()->abort(403);
                        break;
                }

                break;
        }
    }


    /**
     * @param $relationship_model
     * @param $relationship_type
     * @param $remote_relationship_type
     * @param string $kind_of_request
     */
    protected function checkRelationshipAuthorisation($relationship_model, $relationship_type, $remote_relationship_type, $kind_of_request = '' )
    {
        if ( $kind_of_request ){
            $this->checkKindOfRequest($kind_of_request, $relationship_type, $remote_relationship_type);
        }

        switch( $relationship_type ){
            case 'BelongsTo':
            case 'BelongsToMany':
            case 'MorphMany':
            case 'MorphToMany':
                $this->authorizeByPolicy('view', $relationship_model);
                break;

            case 'HasMany':
            case 'HasOne':
                $this->authorizeByPolicy('update', $relationship_model);
                break;

            default:
                $this->authorizeByPolicy('view', $relationship_model);
                break;
        }
    }


    /**
     * @param $modelName
     * @param $relation_type
     * @return string
     */
    private function getRemoteRelationshipType($modelName, $relation_type)
    {
        $relationship_config = config('bagaaravel.jsonapi.' . $relation_type);
        $relationship_model = $this->getRelationshipModel($relation_type);

        $relationship_model_method_name = array_search($modelName, $relationship_config['relationships']);
        $relationship_type = class_basename($relationship_model->{$relationship_model_method_name}());

        return $relationship_type;
    }


    /**
     * @param $modelName
     * @param $relation_name
     * @param $relation_type
     * @return string
     */
    private function getRelationshipType($modelName, $relation_name, $relation_type)
    {
        $own_config = config('bagaaravel.jsonapi.' . $modelName);
        $model = app($own_config['model']);

        if ( $relationship_type = $this->getRemoteRelationshipType($modelName, $relation_type) != 'MorphMany' ){
            $relationship_type = class_basename( $model->{$relation_name}() );
        }

        return $relationship_type;
    }


    /**
     * @param $relation_type
     * @return \Illuminate\Foundation\Application|mixed
     */
    private function getRelationshipModel($relation_type)
    {
        $relationship_config = config('bagaaravel.jsonapi.' . $relation_type);
        $relationship_model = app($relationship_config['model']);

        return $relationship_model;
    }


    /**
     * @param FormRequestInterface $request
     * @param string $kind_of_request
     */
    protected function handleRelationships(FormRequestInterface $request, $kind_of_request = '')
    {
        if ( $relationships = $request->input('data.relationships') ){
            foreach( $relationships as $relationship_name => $relationship_data ){

                if ( ! is_numeric(array_keys($relationship_data['data'])[0]) ) $relationship_data['data'] = [$relationship_data['data']];

                foreach( $relationship_data['data'] as $rdata ){
                    //find type of relationship
                    $relationship_type = $this->getRelationshipType(static::$model, $relationship_name, $rdata['type']);
                    $remote_relationship_type = $this->getRemoteRelationshipType(static::$model, $rdata['type']);

                    //check relationship authorisation
                    $this->checkRelationshipAuthorisation($this->getRelationshipModel($rdata['type'])->find($rdata['id']), $relationship_type, $remote_relationship_type, $kind_of_request);

                    //pass relation to gateway
                    if ( ! isset($rdata['meta']) ) $rdata['meta'] = [];
                    $this->gateway->addRelationshipToSave( $relationship_type, $relationship_name, $rdata['type'], $rdata['id'], $rdata['meta'] );
                }
            }
        }
    }


    /**
     * @param FormRequestInterface|\Illuminate\Http\Request $request
     * @return mixed
     */
    public function store(FormRequestInterface $request)
    {
        $this->authorizeByPolicy('create', app($this->config['model']));
        $this->handleRelationships($request);

        $item = $this->gateway->save($request->input('data.attributes'));

        return $this->responder->transformAndRespondContentCreated($item);
    }

    /**
     * Update the specified resource
     *
     * @param FormRequestInterface $request
     * @param $id
     * @return Response
     * @internal param array $ids
     * @internal param int $id
     */
    public function update(FormRequestInterface $request, $id)
    {
        $this->authorizeByPolicy('update', $this->gateway->getById($id));
        // Disabled until we implement this.    
        //$this->handleRelationships($request, 'updateByResource');

        $item = $this->gateway->update($id, $request->input('data.attributes', []));

        return $this->responder->transformAndRespondItem($item);
    }


    /**
     * @return mixed
     */
    private function getRequestedRelationName()
    {
        $segments = request()->segments();
        $relation = end($segments);

        return $relation;
    }

    /**
     * @param $id
     * @param $relation
     * @return mixed
     */
    protected function getRelated($id, $relation)
    {
        $item = $this->gateway->getById($id);
        $this->authorizeByPolicy('view', $item);

        //check if item is polymorphic
        if ( method_exists($item, class_basename($item) . 'able') ){
            $relatedItem = $item->{class_basename($item) . 'able'};
        } else {
            $relatedItem = $item->{$relation};
        }

        return $relatedItem;
    }

    /**
     * @param $id
     * @return mixed
     */
    public function related_resource_show($id)
    {
        $relation = $this->getRequestedRelationName();
        $relatedItem = $this->getRelated($id, $relation);
        if (!$this->responder->allowRelationshipShow($relation)) {
            abort(401);
        }
        $responder = new JsonApiResponder($this->config['relationships'][$relation]);
        $responder->setRelationshipsEnabled(false);

        if ( is_a($relatedItem, Collection::class) ){
            return $responder->transformAndRespondCollection($relatedItem);
        } else {
            return $responder->transformAndRespondItem($relatedItem);
        }
    }

    /**
     * @param $id
     * @return mixed
     */
    public function relationships_show($id)
    {
        $relation = $this->getRequestedRelationName();
        $relatedItem = $this->getRelated($id, $relation);
        if (!$this->responder->allowRelationshipShow($relation)) {
            abort(401);
        }
        $responder = new JsonApiResponder($this->config['relationships'][$relation]);
        $responder->setRelationshipsResponseEnabled(true);

        if ( is_a($relatedItem, Collection::class) ){
            return $responder->transformAndRespondCollection($relatedItem);
        } else {
            return $responder->transformAndRespondItem($relatedItem);
        }
    }

    public function relationships_update($id)
    {
//        $relation = $this->getRequestedRelationName();
//        $relatedItem = $this->getRelated($id, $relation);

        //authorize update

        //todo implement
    }

}
