<?php

namespace Bagaaravel\Api\Responders;


use Bagaaravel\Api\Transformers\GenericTransformer;
use Bagaaravel\Api\Transformers\TransformerInterface;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Console\AppNamespaceDetectorTrait;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\QueryException;
use Illuminate\Http\Response;
use Illuminate\Support\Collection;
use Illuminate\Validation\ValidationException;
use PhpParser\Node\Scalar\Encapsed;
use Symfony\Component\Config\Definition\Exception\Exception;
use Symfony\Component\Debug\Exception\FatalErrorException;
use Symfony\Component\HttpKernel\Exception\HttpException;

class ApiResponder implements ResponderInterface
{
    use AppNamespaceDetectorTrait;

    /**
     * @var array
     */
    protected $headers = [];

    protected $model;
    protected $config;
    protected $relationships = [];

    /**
     * @var TransformerInterface
     */
    protected $transformer;

    protected $pagingEnabled = false;
    protected $pagingPerPage = 20;

    function __construct($config = null)
    {
        if ($config) {
            $this->setConfig($config);
        }

        $this->setTransformer($this->getAutoDetectedTransformer());
    }

    /**
     * @param mixed $config
     */
    protected function setConfig($config)
    {
        $this->model = $config;
        $this->config = config('bagaaravel.jsonapi.' . $config);
        if (isset($this->config['relationships'])) {
            $this->relationships = $this->config['relationships'];
        }
    }


    /**
     * @return GenericTransformer|\Illuminate\Foundation\Application|mixed
     */
    protected function getAutoDetectedTransformer()
    {
        $transformer = ucfirst($this->model) . 'Transformer';

        if (class_exists($this->getAppNamespace() . "Transformers\\${transformer}")) {
            return app($this->getAppNamespace() . "TransFormers\\${transformer}");
        }

        return app(GenericTransformer::class);
    }

    /**
     * @param TransformerInterface $transformer
     */
    public function setTransformer(TransformerInterface $transformer)
    {
        $this->transformer = $transformer;
        $this->transformer->setConfig($this->model);
    }

    /**
     * @param $headers
     */
    public function setHeaders($headers)
    {
        $this->headers = $headers;
    }

    /**
     * @param $name
     * @param $value
     */
    public function setHeader($name, $value)
    {
        $this->headers[$name] = $value;
    }


    /**
     * @param       $object
     * @param int   $status
     * @param array $headers
     * @return mixed
     */
    protected function respond($object, $status = 200, array $headers = [])
    {
        $headers = array_merge($this->headers, $headers);

        return response()->json($object, $status)->withHeaders($headers);
    }

    /**
     * @param     $item
     * @param int $status
     * @return mixed
     */
    protected function respondItem($item, $status = 200)
    {
        return $this->respond($item, $status);
    }


    /**
     * @param     $item
     * @param int $status
     * @return mixed
     */
    public function transformAndRespondItem($item, $status = 200)
    {
        return $this->respondItem($this->transformer->transform($item), $status);
    }


    /**
     * @param     $collection
     * @param int $status
     * @return mixed
     */
    protected function respondCollection($collection, $status = 200)
    {
        return $this->respond($collection, $status);
    }


    /**
     * @param     $collection
     * @param int $status
     * @return mixed
     */
    public function transformAndRespondCollection($collection, $status = 200)
    {
        if ($this->pagingEnabled) {
            $collection->setCollection(new Collection($this->transformer->transform($collection->getCollection())));

            return $this->respondCollection($collection);
        }

        return $this->respondCollection($this->transformer->transform($collection), $status);
    }


    /**
     * @return mixed
     */
    public function respondNoContent()
    {
        return $this->respond(null, Response::HTTP_NO_CONTENT);
    }

    /**
     * @param $object
     * @param $headers
     * @return mixed
     */
    protected function respondContentCreated($object, $headers = [])
    {
        return $this->respond($object, Response::HTTP_CREATED, $headers);
    }


    /**
     * @param       $object
     * @param array $headers
     * @return mixed
     */
    public function transformAndRespondContentCreated($object, $headers = [])
    {
        return $this->respondContentCreated($this->transformer->transform($object), $headers);
    }

    /**
     * @param boolean $pagingEnabled
     */
    public function setPagingEnabled($pagingEnabled)
    {
        $this->pagingEnabled = $pagingEnabled;
    }

    /**
     * @param int $pagingPerPage
     */
    public function setPagingPerPage($pagingPerPage)
    {
        $this->pagingPerPage = $pagingPerPage;
    }

    public function respondException(\Exception $e)
    {
        $statusCode = Response::HTTP_INTERNAL_SERVER_ERROR;
        if ($e instanceof FatalErrorException) {
            $statusCode = Response::HTTP_INTERNAL_SERVER_ERROR;
        }
        if ($e instanceof AuthenticationException) {
            $statusCode = Response::HTTP_UNAUTHORIZED;
        }
        if ($e instanceof AuthorizationException) {
            $statusCode = Response::HTTP_FORBIDDEN;
        }
        if ($e instanceof ModelNotFoundException) {
            $statusCode = Response::HTTP_NOT_FOUND;
        }
        if ($e instanceof QueryException) {
            $statusCode = Response::HTTP_INTERNAL_SERVER_ERROR;
        }
        if ($e instanceof ValidationException) {
            return $e->response;
        }
        if ($e instanceof HttpException) {
            $statusCode = $e->getStatusCode();
        }
        $headers = ['Content-Type' => 'application/vnd.api+json'];
        $errorObject = [
            'errors' => [
                'status' => $statusCode,
                'title' => Response::$statusTexts[$statusCode],
                'detail' => $e->getMessage()
            ]
        ];

        if (env('APP_DEBUG')) {
            $errorObject['meta'] = [
                'file' => $e->getFile()
            ];
        }

        return $this->respond($errorObject, $statusCode, $headers);
    }

}
