<?php

namespace Bagaaravel\Api;

use Bagaaravel\Api\Requests\FormRequestInterface;
use Bagaaravel\Api\Responders\ResponderInterface;
use Bagaaravel\Models\DummyPolicyModel;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Bagaaravel\Api\Gateways\GenericGateway;

/**
 * Class JsonCrudController
 * @package Bagaaravel\Api
 */
abstract class JsonCrudController extends Controller
{
    use AuthorizesRequests;

    protected $gateway;
    public static $model;
    protected $config;

    protected $pagingEnabled = false;
    protected $pagingPerPage = 20;
    /**
     * @var ResponderInterface
     */
    protected $responder;


    /**
     * JsonCrudController constructor.
     * @param GenericGateway     $gateway
     * @param ResponderInterface $responder
     */
    public function __construct(GenericGateway $gateway, ResponderInterface $responder)
    {
        $this->gateway = $gateway;
        $this->gateway->setConfig(static::$model);

        $this->config = config('bagaaravel.jsonapi.' . static::$model);

        $this->responder = $responder;
        $this->responder->setPagingEnabled($this->pagingEnabled);
        $this->responder->setPagingPerPage($this->pagingPerPage);
    }

    public abstract function defaults();

    /**
     * Display a listing of the resource
     *
     * @return Response
     */
    public function index()
    {
        $this->gateway->setFilters($this->parseFilters());
        $this->gateway->setSorting($this->parseSorting());
        if ($this->pagingEnabled) {
            $items = $this->gateway->getAllPaginated($this->pagingPerPage);
            return $this->responder->transformAndRespondCollection($items);
        }
        $items = $this->gateway->getAll();
        return $this->responder->transformAndRespondCollection($items);
    }

    /**
     * @return array
     */
    protected function parseFilters()
    {
        $filters = [];
        if (isset($this->config['filters'])) {
            foreach (\Request::get('filter', []) as $key => $value) {
                if (in_array($key, $this->config['filters'])) {
                    $filters[$key] = $value;
                }
            }
        }

        if (!empty($this->defaults())) {
            $filters = array_merge($filters, $this->defaults());
        }

        return $filters;
    }

    /**
     * @return array
     */
    protected function parseSorting()
    {
        $sorting = [];
        $sortString = \Request::get('sort', '');
        if ($sortString === '') {
            return $sorting;
        }
        foreach (explode(',', $sortString) as $item) {
            if (strpos($item, '-') === 0) {
                $key = substr($item, 1);
                $direction = 'DESC';
            } else {
                $key = $item;
                $direction = 'ASC';
            }
            $sort['key'] = $key;
            $sort['direction'] = $direction;
            $sorting[] = $sort;
        }
        return $sorting;
    }

    /**
     * @param FormRequestInterface|\Illuminate\Http\Request $request
     * @return mixed
     */
    public function store(FormRequestInterface $request)
    {
        $this->authorizeByPolicy('create', app($this->config['model']));

        $item = $this->gateway->save($request->input());

        return $this->responder->transformAndRespondContentCreated($item);
    }

    /**
     * Display the specified resource
     *
     * @param $id
     * @return Response
     */
    public function show($id)
    {
        $item = $this->gateway->getById($id);
        $this->authorizeByPolicy('view', $item);

        return $this->responder->transformAndRespondItem($item);
    }

    /**
     * Update the specified resource
     *
     * @param FormRequestInterface $request
     * @param                      $id
     * @return Response
     */
    public function update(FormRequestInterface $request, $id)
    {
        $this->authorizeByPolicy('update', $this->gateway->getById($id));

        $item = $this->gateway->update($id, $request->input());

        return $this->responder->transformAndRespondItem($item);
    }

    /**
     * Remove the specified resource
     *
     * @param $id
     * @return Response
     */
    public function destroy($id)
    {
        $item = $this->gateway->getById($id);

        $this->authorizeByPolicy('delete', $item);

        $this->gateway->delete($id);

        return $this->responder->respondNoContent();
    }

    /**
     * @param $action
     * @param $model
     */
    protected function authorizeByPolicy($action, $model)
    {
        $policyExists = false;
        try {
            if (\Gate::getPolicyFor($model)) {
                $policyExists = true;
            }
        } catch (\InvalidArgumentException $e) {
            $policyExists = false;
        }

        if ($policyExists) {
            $this->authorize($action, $model);
        } else {
            //if user is logged in, apply default policy
            if ( \Auth::user() ){
                $this->authorize($action, new DummyPolicyModel);
            }
        }
    }
}
