<?php
namespace Bagaaravel\Api\Responders;

use Bagaaravel\Api\Transformers\GenericJsonApiTransformer;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;

class JsonApiResponder extends ApiResponder implements ResponderInterface
{

    protected $relationshipsEnabled = true;
    protected $relationshipsResponseEnabled = false;

    public function __construct($config = null)
    {
        parent::__construct($config);
        $this->setHeader('Content-Type', 'application/vnd.api+json');
    }

    /**
     * @param boolean $relationshipsEnabled
     */
    public function setRelationshipsEnabled($relationshipsEnabled)
    {
        $this->relationshipsEnabled = $relationshipsEnabled;
    }

    /**
     * @param boolean $relationshipsResponseEnabled
     */
    public function setRelationshipsResponseEnabled($relationshipsResponseEnabled)
    {
        $this->relationshipsResponseEnabled = $relationshipsResponseEnabled;
    }

    /**
     * @return GenericJsonApiTransformer|\Illuminate\Foundation\Application|mixed
     */
    protected function getAutoDetectedTransformer()
    {
        $transformer = ucfirst($this->model) . 'Transformer';

        if (class_exists($this->getAppNamespace() . "Transformers\\${transformer}")) {
            return app($this->getAppNamespace() . "TransFormers\\${transformer}");
        }

        return app(GenericJsonApiTransformer::class);
    }

    /**
     * @return array
     */
    private function createLinks($collection = null)
    {
        $links = [
            'self' => \Request::fullUrl()
        ];
        if ($this->pagingEnabled && $collection instanceof LengthAwarePaginator) {
            $links['previous'] = $collection->previousPageUrl();
            $links['next'] = $collection->nextPageUrl();
            $links['first'] = $collection->url(1);
            $links['last'] = $collection->url($collection->lastPage());
        }

        return $links;
    }

    /**
     * @return array
     */
    private function createMeta($collection = null)
    {
        $meta = [];
        if ($this->pagingEnabled && $collection instanceof LengthAwarePaginator) {
            $meta['total'] = $collection->total();
            $meta['per_page'] = $collection->perPage();
        }
        return $meta;
    }

    private function createRelationshipMeta($item)
    {
        $meta = null;
        if (isset($item->toArray()['pivot'])) {
            //are there keys that don't end in _id -> add to meta data
            $meta = [];
            foreach ($item->toArray()['pivot'] as $key => $value) {
                if (substr($key, strlen($key) - 3, 3) !== '_id') {
                    $meta[$key] = $value;
                }
            }
        }
        return $meta;
    }


    /**
     * @param $item
     * @param $cb
     */
    private function foreachRelationship($item, $cb)
    {
        //check if item is polymorphic
        $isPolymorphic = false;
        if (method_exists($item, camel_case($this->model) . 'able')) {
            $isPolymorphic = true;
        }

        foreach ($this->relationships as $name => $model) {
            if ($isPolymorphic) {
                $relation = $item->{camel_case($this->model) . 'able'};
            } else {
                $relation = $item->{$name};
            }

            $cb($name, $relation);
        }
    }

    /**
     * @param $item
     * @return array
     */
    private function createRelationshipData($item)
    {
        $data = [
            'type' => class_basename($item->getModel()),
            'id' => $item->getKey()
        ];
        if ($meta = $this->createRelationshipMeta($item)) {
            $data['meta'] = $meta;
        }

        return $data;
    }


    /**
     * @param $relation
     * @return array|null|void
     */
    private function createRelationship($relation)
    {
        $rdata = null;

        if (is_a($relation, Collection::class)) {
            if (!$relation->count()) {
                return;
            }

            $rdata = $relation->map(function ($item) {
                $data = $this->createRelationshipData($item);
                return $data;
            });

        } else {
            if (!$relation) {
                return;
            }
            $rdata = $this->createRelationshipData($relation);
        }

        return $rdata;
    }

    /**
     * @param $relation
     * @return array
     */
    public function allowRelationshipShow($relation)
    {
        // not applicable when not logged in
        if (! auth()->check()) {
            return true;
        }
        $allowedRelations = [];
        if (!isset($this->config['visible_relationships'])) {
            $allowedRelations = array_keys($this->config['relationships']);
        } elseif (!auth()->user()->roles()->count() && isset($this->config['visible_relationships']['default'])) {
            $allowedRelations = $this->config['visible_relationships']['default'];
        } else {
            $roles = $this->config['visible_relationships'];
            foreach ($roles as $role => $relations) {
                if (auth()->user()->hasRole($role)) {
                    $allowedRelations = $relations;
                }
            }
        }
        return in_array($relation, $allowedRelations, null);
    }

    /**
     * @param $item
     * @return null
     */
    private function createRelationShips($item)
    {
        $list = null;

        if (!$this->relationshipsEnabled) {
            return $list;
        }

        $this->foreachRelationship($item, function ($relationName, $relation) use (&$list) {
            $relationshipData = $this->createRelationship($relation);
            if ($relationshipData && $this->allowRelationshipShow($relationName)) {
                $list[$relationName]['data'] = $relationshipData;
            }
        });

        return $list;
    }

    /**
     * @param $item
     * @return array
     */
    private function createDataItem($item)
    {
        if ($this->relationshipsResponseEnabled) {
            $dataItem = $this->createRelationshipData($item);
        } else {
            $dataItem = [
                'id' => $item->getKey(),
                'type' => class_basename($item->getModel()),
                'attributes' => $this->transformer->transformItem($item)
            ];
            $relationships = $this->createRelationShips($item);
            if (!empty($relationships)) {
                $dataItem['relationships'] = $relationships;
            }
        }

        return $dataItem;
    }

    /**
     * @param $collection
     * @return array
     */
    private function createData($collection)
    {
        if ($this->pagingEnabled && method_exists($collection, 'getCollection')) {
            $collection = $collection->getCollection();
        }
        if (is_a($collection, Collection::class)) {
            $data = [];
            foreach ($collection as $item) {
                $data[] = $this->createDataItem($item);
            }
        } else {
            $data = $this->createDataItem($collection);
        }

        return $data;
    }

    /**
     * @param     $collection
     * @param int $status
     * @return mixed
     */
    public function transformAndRespondCollection($collection, $status = 200)
    {
        $return = [
            'data' => $this->createData($collection),
            'links' => $this->createLinks($collection),
        ];
        $meta = $this->createMeta($collection);
        if (!empty($meta)) {
            $return['meta'] = $meta;
        }
        $included = $this->createIncluded($collection);
        if (!empty($included)) {
            $return['included'] = $included;
        }

        return $this->respondCollection($return, $status);
    }


    /**
     * @param     $item
     * @param int $status
     * @return mixed
     */
    public function transformAndRespondItem($item, $status = 200)
    {
        $return = [
            'data' => $this->createData($item),
            'links' => $this->createLinks()
        ];
        $meta = $this->createMeta();
        if (!empty($meta)) {
            $return['meta'] = $meta;
        }
        $included = $this->createIncluded($item);
        if (!empty($included)) {
            $return['included'] = $included;
        }

        return $this->respondItem($return, $status);
    }

    /**
     * @param       $object
     * @param array $headers
     * @return mixed
     */
    public function transformAndRespondContentCreated($object, $headers = [])
    {
        $return = [
            'data' => $this->createData($object),
            'links' => $this->createLinks()
        ];
        $meta = $this->createMeta();
        if (!empty($meta)) {
            $return['meta'] = $meta;
        }

        $headers['Location'] = route($this->getBaseRouteName() . '.show', [$object->getKey()]);

        return $this->respondContentCreated($return, $headers);
    }


    /**
     * @return string
     */
    protected function getBaseRouteName()
    {
        $name = request()->route()->getName();
        $nameA = explode('.', $name);
        array_pop($nameA);

        return implode('.', $nameA);
    }

    protected function createIncluded($collection)
    {
        $included = [];
        $includeString = \Request::get('include', '');
        if ($includeString === '') {
            return $included;
        }
        $collection = $collection instanceof Collection ? $collection : [$collection];
        foreach (explode(',', $includeString) as $relation) {
            foreach ($collection as $item) {
                $polyRelation = camel_case(class_basename($item)) . 'able';
                if (method_exists($item, $relation) || method_exists($item, $polyRelation)) {
                    $responder = new JsonApiResponder($this->config['relationships'][$relation]);
                    if ($item->{$relation} instanceof Collection) {
                        $included = array_merge($included, $responder->createData($item->{$relation}));
                    }
                    if ($item->{$relation} instanceof $responder->config['model']) {
                        $included[] = $responder->createData($item->{$relation});
                    }
                    if ($item->{$polyRelation} instanceof $responder->config['model']) {
                        $included[] = $responder->createData($item->{$polyRelation});
                    }
                }
            }
        }

        return $included;
    }
}
