<?php
namespace Bagaaravel\Api\Responders;

use Bagaaravel\Api\Transformers\GenericJsonApiTransformer;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;

class JsonApiResponder extends ApiResponder implements ResponderInterface
{

    protected $relationshipsEnabled = true;
    protected $relationshipsResponseEnabled = false;

    protected $createResourceAndRelationshipLinks = true;

    protected $relationshipsAsLinks = false;

    public function __construct($config = null)
    {
        parent::__construct($config);
        $this->setHeader('Content-Type', 'application/vnd.api+json');
    }

    /**
     * @param boolean $relationshipsEnabled
     */
    public function setRelationshipsEnabled($relationshipsEnabled)
    {
        $this->relationshipsEnabled = $relationshipsEnabled;
    }

    /**
     * @param boolean $relationshipsResponseEnabled
     */
    public function setRelationshipsResponseEnabled($relationshipsResponseEnabled)
    {
        $this->relationshipsResponseEnabled = $relationshipsResponseEnabled;
    }


    /**
     * @param boolean $createResourceAndRelationshipLinks
     */
    public function setCreateResourceAndRelationshipLinks($createResourceAndRelationshipLinks)
    {
        $this->createResourceAndRelationshipLinks = $createResourceAndRelationshipLinks;
    }

    /**
     * @param boolean $relationshipsAsLinks
     */
    public function setRelationshipsAsLinks($relationshipsAsLinks)
    {
        $this->relationshipsAsLinks = $relationshipsAsLinks;
    }

    /**
     * @return GenericJsonApiTransformer|\Illuminate\Foundation\Application|mixed
     */
    protected function getAutoDetectedTransformer()
    {
        $transformer = ucfirst($this->model) . 'Transformer';

        if (class_exists($this->getAppNamespace() . "Transformers\\${transformer}")) {
            return app($this->getAppNamespace() . "TransFormers\\${transformer}");
        }

        return app(GenericJsonApiTransformer::class);
    }

    /**
     * @return array
     */
    private function createLinks($collection = null)
    {
        $links = [
            'self' => \Request::fullUrl()
        ];
        if ($this->pagingEnabled && $collection instanceof LengthAwarePaginator) {
            $links['previous'] = $collection->previousPageUrl();
            $links['next'] = $collection->nextPageUrl();
            $links['first'] = $collection->url(1);
            $links['last'] = $collection->url($collection->lastPage());
        }

        return $links;
    }

    /**
     * @return array
     */
    private function createMeta($collection = null)
    {
        $meta = [];
        if ($this->pagingEnabled && $collection instanceof LengthAwarePaginator) {
            $meta['total'] = $collection->total();
            $meta['per_page'] = $collection->perPage();
        }
        return $meta;
    }

    private function createRelationshipMeta($item)
    {
        $meta = null;
        if (isset($item->pivot)) {
            //are there keys that don't end in _id -> add to meta data
            $meta = [];
            foreach ($item->pivot->toArray() as $key => $value) {
                if (substr($key, strlen($key) - 3, 3) !== '_id') {
                    $meta[$key] = $value;
                }
            }
        }
        return $meta;
    }


    /**
     * @param $item
     * @param $cb
     */
    private function foreachRelationship($item, $cb)
    {
        //check if item is polymorphic
        $isPolymorphic = false;
        if (method_exists($item, camel_case($this->model) . 'able')) {
            $isPolymorphic = true;
        }

        foreach ($this->relationships as $name => $model) {
            if ($isPolymorphic) {
                $relation = $item->{camel_case($this->model) . 'able'};
            } else {
                $relation = $item->{$name};
            }

            $cb($name, $relation);
        }
    }

    /**
     * @param $item
     * @return array
     */
    private function createRelationshipData($item)
    {
        $data = [
            'type' => class_basename($item->getModel()),
            'id' => $item->getKey()
        ];
        if ($meta = $this->createRelationshipMeta($item)) {
            $data['meta'] = $meta;
        }

        return $data;
    }


    /**
     * @param $relation
     * @return array|null|void
     */
    private function createRelationship($relation)
    {
        $rdata = null;

        if (is_a($relation, Collection::class)) {
            if (!$relation->count()) {
                return;
            }

            if ( $this->relationshipsAsLinks ){
                return true;
            }

            $rdata = $relation->map(function ($item) {
                $data = $this->createRelationshipData($item);
                return $data;
            });

        } else {
            if (!$relation) {
                return;
            }

            if ( $this->relationshipsAsLinks ){
                return true;
            }

            $rdata = $this->createRelationshipData($relation);
        }

        return $rdata;
    }

    /**
     * @param $relation
     * @return array
     */
    public function allowRelationshipShow($relation)
    {
        // No visible relations defined
        if (!isset($this->config['visible_relationships'])) {
            return array_key_exists($relation, $this->config['relationships']);
        }
        $roles = $this->config['visible_relationships'];
        // Not logged in or no role and default defined
        if (isset($roles['default']) && (auth()->guest() || !count(auth()->user()->roles))) {
            return in_array($relation, $this->config['visible_relationships']['default']);
        }
        // Not logged or no role in and no default defined
        if (auth()->guest() || !auth()->user()->roles) {
            return array_key_exists($relation, $this->config['relationships']);
        }
        $userRoles = auth()->user()->roles->pluck('slug')->all();
        $matchedRoles = array_intersect($userRoles, array_keys($roles));
        // User role is not in config and default is set
        if (isset($roles['default']) && count($matchedRoles) === 0) {
            return in_array($relation, $this->config['visible_relationships']['default']);
        }
        // User role is not in config and default is not set
        if (count($matchedRoles) === 0) {
            return array_key_exists($relation, $this->config['relationships']);
        }
        // User role is in config
        $allowedRelations = [];
        foreach ($matchedRoles as $matchedRole) {
            $allowedRelations = array_merge($allowedRelations, $this->config['visible_relationships'][$matchedRole]);
        }
        return in_array($relation, $allowedRelations);
    }

    /**
     * @param $item
     * @return null
     */
    private function createRelationShips($item)
    {
        $list = null;

        if (!$this->relationshipsEnabled) {
            return $list;
        }

        $this->foreachRelationship($item, function ($relationName, $relation) use (&$list, $item) {
            $relationshipData = $this->createRelationship($relation);
            if ($relationshipData && $this->allowRelationshipShow($relationName)) {

                if ( ! $this->relationshipsAsLinks ){
                    $list[$relationName]['data'] = $relationshipData;
                }

                if ( $this->createResourceAndRelationshipLinks ){
                    if ( \Route::has($this->getBaseRouteName() . '.relation.' . $relationName . '.show') ){
                        $list[$relationName]['links'] = [
                            'self' => route($this->getBaseRouteName() . '.relation.' . $relationName . '.show', ['id' => $item->getKey()]),
                            'related' => route($this->getBaseRouteName() . '.related.' . $relationName . '.show', ['id' => $item->getKey()])
                        ];
                    }
                }
            }
        });

        return $list;
    }

    /**
     * @param $item
     * @return array
     */
    private function createDataItem($item)
    {
        $item = $this->setSparseFields($item);
        if ($this->relationshipsResponseEnabled) {
            $dataItem = $this->createRelationshipData($item);
        } else {
            $dataItem = [
                'id' => $item->getKey(),
                'type' => class_basename($item->getModel()),
                'attributes' => $this->transformer->transformItem($item)
            ];

            if ( $this->createResourceAndRelationshipLinks ){
                $dataItem['links'] = [
                    'self' => route($this->getBaseRouteName() . '.show', ['id' => $item->getKey()])
                ];
            }

            $relationships = $this->createRelationShips($item);
            if (!empty($relationships)) {
                $dataItem['relationships'] = $relationships;
            }
        }

        return $dataItem;
    }

    /**
     * @param $collection
     * @return array
     */
    private function createData($collection)
    {
        if ($this->pagingEnabled && method_exists($collection, 'getCollection')) {
            $collection = $collection->getCollection();
        }
        if (is_a($collection, Collection::class)) {
            $data = [];
            foreach ($collection as $item) {
                $data[] = $this->createDataItem($item);
            }
        } else {
            $data = $collection ? $this->createDataItem($collection) : null;
        }

        return $data;
    }

    /**
     * @param     $collection
     * @param int $status
     * @return mixed
     */
    public function transformAndRespondCollection($collection, $status = 200)
    {
        $return = [
            'data' => $this->createData($collection),
            'links' => $this->createLinks($collection),
        ];
        $meta = $this->createMeta($collection);
        if (!empty($meta)) {
            $return['meta'] = $meta;
        }
        $included = $this->createIncluded($collection);
        if (!empty($included)) {
            $return['included'] = $included;
        }

        return $this->respondCollection($return, $status);
    }


    /**
     * @param     $item
     * @param int $status
     * @return mixed
     */
    public function transformAndRespondItem($item, $status = 200)
    {
        $return = [
            'data' => $this->createData($item),
            'links' => $this->createLinks()
        ];
        $meta = $this->createMeta();
        if (!empty($meta)) {
            $return['meta'] = $meta;
        }
        $included = $this->createIncluded($item);
        if (!empty($included)) {
            $return['included'] = $included;
        }

        return $this->respondItem($return, $status);
    }

    /**
     * @param       $object
     * @param array $headers
     * @return mixed
     */
    public function transformAndRespondContentCreated($object, $headers = [])
    {
        $return = [
            'data' => $this->createData($object),
            'links' => $this->createLinks()
        ];
        $meta = $this->createMeta();
        if (!empty($meta)) {
            $return['meta'] = $meta;
        }

        $headers['Location'] = route($this->getBaseRouteName() . '.show', [$object->getKey()]);

        return $this->respondContentCreated($return, $headers);
    }


    /**
     * @return string
     */
    protected function getBaseRouteName()
    {
        $name = request()->route()->getName();
        $nameA = explode('.', $name);
        array_pop($nameA);

        return implode('.', $nameA);
    }

    protected function createIncluded($collection)
    {
        $included = [];
        $includeString = \Request::get('include', '');
        if ($includeString === '') {
            return $included;
        }
        $collection = is_iterable($collection) ? $collection : [$collection];
        foreach (explode(',', $includeString) as $relation) {
            $this->processIncludedRelations($collection, $relation, $included);
        }

        $unique_keys = [];

        return array_filter($included, function ($item) use (&$unique_keys) {
            $key = $item['type'] . ':' . $item['id'];
            $exists = in_array($key, $unique_keys);

            if (!$exists) {
                $unique_keys[] = $key;
            }

            return !$exists;
        });
    }

    protected function processIncludedRelations($collection, $relation_path, &$included)
    {
        $relation_parts = explode('.', $relation_path, 2);

        if ( isset($relation_parts[0]) ) {
            $relation = $relation_parts[0];
            $responder = null;

            $models = new Collection();
            foreach ($collection as $item) {
                $item = $this->setSparseFields($item);
                $polyRelation = camel_case(class_basename($item)) . 'able';
                if (method_exists($item, $relation) || method_exists($item, $polyRelation)) {
                    $responder = new JsonApiResponder($this->config['relationships'][$relation]);
                    $responder->setCreateResourceAndRelationshipLinks(false);

                    if ($item->{$relation} instanceof Collection) {
                        $included = array_merge($included, $responder->createData($item->{$relation}));
                        $models = $models->merge($item->{$relation});
                    }
                    if ($item->{$relation} instanceof $responder->config['model']) {
                        $included[] = $responder->createData($item->{$relation});
                        $models[] = $item->{$relation};
                    }
                    if ($item->{$polyRelation} instanceof $responder->config['model']) {
                        $included[] = $responder->createData($item->{$polyRelation});
                        $models[] = $item->{$polyRelation};
                    }
                }
            }

            if ( $responder && isset($relation_parts[1]) ) {
                $responder->processIncludedRelations($models, $relation_parts[1], $included);
            }
        }
    }
    /**
     * @param $item
     * @return mixed
     */
    protected function setSparseFields($item)
    {
        if ($fieldsets = \Request::get('fields', null)) {
            foreach ($fieldsets as $model => $fields) {
                if (ucfirst($model) === class_basename($item)) {
                    $attributes = array_keys($item->getAttributes());
                    $visible = array_filter($attributes, function ($attribute) use ($fields) {
                        return $attribute == 'id' || in_array($attribute, explode(',', $fields));
                    });
                    $item->makeHidden(array_diff($attributes, $visible));
                }
            }
        }

        return $item;
    }
}
