<?php
namespace Bagaaravel\Api;

use Bagaaravel\Api\Gateways\GenericGateway;
use Bagaaravel\Api\Requests\FormRequestInterface;
use Bagaaravel\Api\Responders\JsonApiResponder;
use Bagaaravel\Api\Responders\ResponderInterface;
use Bagaaravel\Utils\ConfigHandler;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;

abstract class JsonApiController extends JsonCrudController
{

    protected $relationshipPagingEnabled = false;
    protected $relationshipPagingPerPage = 20;

    protected $relationshipsAsLinks = false;

    /**
     * JsonApiController constructor.
     * @param GenericGateway $gateway
     * @param ResponderInterface $responder
     */
    function __construct(GenericGateway $gateway, ResponderInterface $responder)
    {
        parent::__construct($gateway, $responder);
        $responder->setRelationshipsAsLinks($this->relationshipsAsLinks);
    }


    /**
     * @param $kind_of_request
     * @param $relationship_type
     * @param $remote_relationship_type
     */
    protected function checkKindOfRequest($kind_of_request, $relationship_type, $remote_relationship_type)
    {
        switch ( $kind_of_request ){
            case 'createRelation':
            case 'deleteRelation':
                if ( $relationship_type == 'BelongsToMany' && $remote_relationship_type == 'BelongsToMany' ){
                    //ok
                } else {
                    app()->abort(403);
                }
                break;

            case 'updateByResource':

                switch ($relationship_type){
                    case 'HasMany':
                    case 'BelongsToMany':
                    case 'MorphMany':
                    case 'MorphToMany':
                        app()->abort(403);
                        break;
                    case 'BelongsTo':
                        if ( $remote_relationship_type == 'HasOne' ) app()->abort(403);
                        break;

                    case 'HasOne':
                        if ( $remote_relationship_type == 'BelongsTo' ) app()->abort(403);
                        break;
                }

                break;

            case 'updateByRelationship':
                if ($relationship_type == 'HasOne' && $remote_relationship_type == 'BelongsTo') app()->abort(403);
                break;
        }
    }


    /**
     * @param $relationship_model
     * @param $relationship_type
     * @param $remote_relationship_type
     * @param string $kind_of_request
     */
    protected function checkRelationshipAuthorisation($model, $relationship_name, $relationship_model, $relationship_type, $remote_relationship_type, $kind_of_request = '' )
    {
        if ( $kind_of_request ){
            $this->checkKindOfRequest($kind_of_request, $relationship_type, $remote_relationship_type);
        }

        if ( $this->isPolicyForRelationExists($model, $relationship_name) ) {
            $this->authorize($relationship_name, [$model, $relationship_model]);
            return;
        }

        switch( $relationship_type ){
            case 'BelongsTo':
            case 'BelongsToMany':
            case 'MorphMany':
            case 'MorphToMany':
                $this->authorizeByPolicy('view', $relationship_model);
                break;

            case 'HasMany':
            case 'HasOne':
                $this->authorizeByPolicy('update', $relationship_model);
                break;

            default:
                $this->authorizeByPolicy('view', $relationship_model);
                break;
        }
    }

    protected function isPolicyForRelationExists($model, $relationship_name)
    {
        $policyExists = false;
        try {
            if (\Gate::getPolicyFor($model)) {
                $policyExists = true;
            }
        } catch (\InvalidArgumentException $e) {
            $policyExists = false;
        }

        $actionExists = false;
        if ( $policyExists ){
            $policy = \Gate::getPolicyFor($model);
            if ( method_exists($policy, $relationship_name) ){
                $actionExists = true;
            }
        }

        return $policyExists && $actionExists;
    }


    /**
     * @param $modelName
     * @param $relation_type
     * @return string
     */
    private function getRemoteRelationshipType($modelName, $relation_type)
    {
        $relationship_config = config('bagaaravel.jsonapi.' . $relation_type);
        $relationship_model = $this->getRelationshipModel($relation_type);
        $relationship_model_method_name = array_search($modelName, ConfigHandler::extractRelationships($relationship_config['relationships']));
        $relationship_type = class_basename($relationship_model->{$relationship_model_method_name}());

        return $relationship_type;
    }
    


    /**
     * @param $modelName
     * @param $relation_name
     * @param $relation_type
     * @return string
     */
    private function getRelationshipType($modelName, $relation_name, $relation_type)
    {
        $own_config = config('bagaaravel.jsonapi.' . $modelName);
        $model = app($own_config['model']);

        if ( $relationship_type = $this->getRemoteRelationshipType($modelName, $relation_type) != 'MorphMany' ){
            $relationship_type = class_basename( $model->{$relation_name}() );
        }

        return $relationship_type;
    }


    /**
     * @param $relation_type
     * @return \Illuminate\Foundation\Application|mixed
     */
    private function getRelationshipModel($relation_type)
    {
        $relationship_config = config('bagaaravel.jsonapi.' . $relation_type);
        $relationship_model = app($relationship_config['model']);

        return $relationship_model;
    }


    /**
     * @param $relationship_name
     * @param $relationship_data
     * @param string $kind_of_request
     */
    protected function handleRelationship($relationship_name, $relationship_data, $kind_of_request = '', $id = null)
    {
        // First delete existing relations for many-to-many
        $model = $id ? app(config('bagaaravel.jsonapi.' . static::$model . '.model'))->find($id) : app(config('bagaaravel.jsonapi.' . static::$model . '.model'));
        if ($kind_of_request == 'updateByRelationship' && $id) {
            $related_items = $model->{$relationship_name}()->get();
            foreach ($related_items as $related_item) {
                $type = ConfigHandler::extractRelationType($this->config['relationships'][$relationship_name]);
                $relationship_type = $this->getRelationshipType(static::$model, $relationship_name, $type);
                $remote_relationship_type = $this->getRemoteRelationshipType(static::$model, $type);
                $this->checkRelationshipAuthorisation($model, $relationship_name, $related_item, $relationship_type, $remote_relationship_type, $kind_of_request);
                // Patching null on one-to-one is still forbidden    
                if ($relationship_type == 'BelongsTo' && $remote_relationship_type == 'HasOne' && is_null($relationship_data['data'])) {
                    abort(403);
                }
                // Prevent dissociation of relation when patching one-to-many
                if ($relationship_type == 'BelongsTo' && !is_null($relationship_data['data'])) {
                    continue;
                }
                $this->gateway->addRelationshipToDelete($relationship_type, $relationship_name, $type, $related_item->id, []);
            }
        }

        // Request is only for deleting relationships
        if (is_null($relationship_data['data']) || $relationship_data['data'] === []) {
            return;
        }

        if (! is_numeric(array_keys($relationship_data['data'])[0])) {
            $relationship_data['data'] = [$relationship_data['data']];
        }

        foreach( $relationship_data['data'] as $rdata ){
            $type = ConfigHandler::extractRelationType($this->config['relationships'][$relationship_name]);
            $relationship_type = $this->getRelationshipType(static::$model, $relationship_name, $type);
            $remote_relationship_type = $this->getRemoteRelationshipType(static::$model, $type);
            $this->checkRelationshipAuthorisation($model, $relationship_name, $this->getRelationshipModel($rdata['type'])->find($rdata['id']), $relationship_type, $remote_relationship_type, $kind_of_request);

            //pass relation to gateway
            if ( ! isset($rdata['meta']) ) $rdata['meta'] = [];

            if ($kind_of_request == 'deleteRelation') {
                $this->gateway->addRelationshipToDelete( $relationship_type, $relationship_name, $rdata['type'], $rdata['id'], $rdata['meta'] );
            } else {
                $this->gateway->addRelationshipToSave( $relationship_type, $relationship_name, $rdata['type'], $rdata['id'], $rdata['meta'] );
            }
        }
    }


    /**
     * @param FormRequestInterface $request
     * @param string $kind_of_request
     */
    protected function handleRelationships(FormRequestInterface $request, $kind_of_request = '', $id)
    {
        if ( $relationships = $request->input('data.relationships') ){
            foreach( $relationships as $relationship_name => $relationship_data ){
                $this->handleRelationship($relationship_name, $relationship_data, $kind_of_request, $id);
            }
        }
    }


    /**
     * @param FormRequestInterface $request
     * @param string $kind_of_request
     */
    protected function handleRelationshipsForRelationRequest(FormRequestInterface $request, $kind_of_request = '', $id = null)
    {
        $relationship_name = $this->getRequestedRelationName();
        if (in_array($kind_of_request, ['updateByRelationship', 'deleteRelation', 'createRelation'])) {
            $model = $this->gateway->getById($id);
            if (!$this->responder->allowRelationshipWritablePermission($relationship_name, $model)) {
                abort(403);
            }
        }
        $relationship_data = $request->input();
        if ($request->has('data') || $request->get('data') === null) {
            $this->handleRelationship($relationship_name, $relationship_data, $kind_of_request, $id);
        }
    }


    /**
     * @param FormRequestInterface|\Illuminate\Http\Request $request
     * @return mixed
     */
    public function store(FormRequestInterface $request)
    {
        $this->authorizeByPolicy('create', app($this->config['model']));
        $this->handleRelationships($request, null, null);

        $item = $this->gateway->save($request->input('data.attributes', []));

        return $this->responder->transformAndRespondContentCreated($item);
    }

    /**
     * Update the specified resource
     *
     * @param FormRequestInterface $request
     * @param $id
     * @return Response
     * @internal param array $ids
     * @internal param int $id
     */
    public function update(FormRequestInterface $request, $id)
    {
        $this->authorizeByPolicy('update', $this->gateway->getById($id));
        $this->handleRelationships($request, 'updateByResource', $id);

        $item = $this->gateway->update($id, $request->input('data.attributes', []));

        return $this->responder->transformAndRespondItem($item);
    }


    /**
     * @return mixed
     */
    private function getRequestedRelationName()
    {
        $segments = request()->segments();
        $relation = end($segments);

        return $relation;
    }

    /**
     * @param $id
     * @param $relation
     * @return mixed
     */
    protected function getRelated($id, $relation)
    {
        $item = $this->gateway->getById($id);
        $this->authorizeByPolicy('view', $item);

        //check if item is polymorphic
        if ( method_exists($item, class_basename($item) . 'able') ){
            $relatedQuery = $item->{class_basename($item) . 'able'}();
        } else {
            $relatedQuery = $item->{$relation}();
        }

        if ( is_a($relatedQuery, HasOne::class) || is_a($relatedQuery, BelongsTo::class) ){
            return $relatedQuery->first();
        }

        $relation_config = config('bagaaravel.jsonapi.' . $this->config['relationships'][$relation]);
        if ( isset($relation_config['gateway']) ){
            $relationGateway = app($relation_config['gateway']);
        } else {
            $relationGateway = app(GenericGateway::class);
        }

        $relationGateway->setConfig($this->config['relationships'][$relation]);
        $relationGateway->setFilters($this->parseFilters($relation_config));
        $relationGateway->setSorting($this->parseSorting());
        if ( $this->relationshipPagingEnabled ){
            $relatedItem = $relationGateway->getAllPaginated($this->parseRelationshipPaging(), $relatedQuery);
            return $relatedItem;
        }

        $relatedItem = $relationGateway->getAll($relatedQuery);
        return $relatedItem;
    }

    /**
     * @param $id
     * @return mixed
     */
    public function related_resource_show($id)
    {
        $relation = $this->getRequestedRelationName();
        $relatedItem = $this->getRelated($id, $relation);
        if (!$this->responder->allowRelationshipShow($relation)) {
            abort(401);
        }
        $model = $this->gateway->getById($id);
        if (!$this->responder->allowRelationshipReadablePermission($relation, $model)) {
            abort(403);
        }
        $responder = new JsonApiResponder($this->config['relationships'][$relation]);
        $responder->setRelationshipsEnabled(true);
        $responder->setCreateResourceAndRelationshipLinks(false);
        $responder->setPagingEnabled($this->relationshipPagingEnabled);
        $responder->setPagingPerPage($this->parseRelationshipPaging());

        if ( is_a($relatedItem, Collection::class) || is_a($relatedItem, LengthAwarePaginator::class) ){
            return $responder->transformAndRespondCollection($relatedItem);
        } else {
            return $responder->transformAndRespondItem($relatedItem);
        }
    }

    /**
     * @param $id
     * @return mixed
     */
    public function relationships_show($id)
    {
        $relation = $this->getRequestedRelationName();
        $relatedItem = $this->getRelated($id, $relation);
        if (!$this->responder->allowRelationshipShow($relation)) {
            abort(401);
        }
        $model = $this->gateway->getById($id);
        if (!$this->responder->allowRelationshipReadablePermission($relation, $model)) {
            abort(403);
        }
        $responder = new JsonApiResponder($this->config['relationships'][$relation]);
        $responder->setRelationshipsResponseEnabled(true);
        $responder->setCreateResourceAndRelationshipLinks(false);
        $responder->setPagingEnabled($this->relationshipPagingEnabled);
        $responder->setPagingPerPage($this->parseRelationshipPaging());

        if ( is_a($relatedItem, Collection::class) || is_a($relatedItem, LengthAwarePaginator::class)){
            return $responder->transformAndRespondCollection($relatedItem);
        } else {
            return $responder->transformAndRespondItem($relatedItem);
        }
    }


    /**
     * @param FormRequestInterface $request
     * @param $id
     * @return mixed
     */
    public function relationships_store(FormRequestInterface $request, $id)
    {
        $this->authorizeByPolicy('update', $this->gateway->getById($id));
        $this->handleRelationshipsForRelationRequest($request, 'createRelation', $id);

        $item = $this->gateway->update($id, []);

        return $this->responder->transformAndRespondContentCreated($item);
    }


    public function relationships_delete(FormRequestInterface $request, $id)
    {
        $this->authorizeByPolicy('update', $this->gateway->getById($id));
        $this->handleRelationshipsForRelationRequest($request, 'deleteRelation', $id);

        $this->gateway->deleteRelationships($id);

        return $this->responder->respondNoContent();
    }


    public function relationships_update(FormRequestInterface $request, $id)
    {
        $this->authorizeByPolicy('update', $this->gateway->getById($id));
        $this->handleRelationshipsForRelationRequest($request, 'updateByRelationship', $id);
        $this->gateway->updateRelationShips($id);
        $item = $this->gateway->getById($id);

        return $this->responder->respondNoContent();
    }


    protected function parseRelationshipPaging()
    {
        $per_page = request('per_page');

        return is_numeric($per_page) && $per_page > 0 ? request('per_page') : $this->relationshipPagingPerPage;
    }



}
