<?php

namespace Bagaaravel\Api\Repositories;

use Bagaaravel\Utils\BagaaravelConfig;
use Bagaaravel\Utils\ConfigHandler;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasOneOrMany;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Database\Query\Builder;

/**
 * Class GenericRepository
 * @package Bagaaravel\Api\Repositories
 */
class GenericRepository implements RepositoryInterface
{
    use FilterTrait;

    /**
     * @var \Illuminate\Database\Eloquent\Model $model
     */
    protected $model;
    /**
     * This is the reference to the resource in bagaaravel.php
     * @var string $modelKey
     */
    protected $modelKey;
    /**
     * @var BagaaravelConfig $bagaaravelConfig
     */
    protected $bagaaravelConfig;
    protected $sorting = [];

    /**
     * @param $model
     */
    public function setModel($model)
    {
        $this->model = app($model);
        $this->bagaaravelConfig = \App::make(BagaaravelConfig::class);
    }

    /**
     * This is the reference to the resource in bagaaravel.php
     * @param string $modelKey
     */
    public function setModelKey($modelKey)
    {
        $this->modelKey = $modelKey;
    }

    /**
     * @param $sorting
     */
    public function setSorting($sorting)
    {
        $this->sorting = $sorting;
    }

    public function applySorting($model)
    {
        if ($model instanceof Relation && request()->has('sort')) {
            $model->getQuery()->getQuery()->orders = [];
        }
        foreach ($this->sorting as $sortIndex => $sort) {

            if (strpos($sort['key'], '.') > -1) {

                [$relation, $param] = explode('.', $sort['key']);
                $relationModel = $model->getModel()->{$relation}();
                $relationshipType = class_basename($relationModel);

                //whitelisting the param name and available relation logic
                if (!\Schema::hasColumn($relationModel->getRelated()->getTable(), $param)) {
                    continue;
                };

                $aggregate = $sort['direction'] == 'DESC' ? 'max' : 'min';

                /** @var $subQuery Builder */
                $subQuery = $this->model
                    ->select($model->getModel()->getTable() . '.id AS id')
                    ->from($model->getModel()->getTable());



                switch ($relationshipType) {
                    case 'HasMany':
                    case 'HasOne':
                        /**
                         * @var $relationModel HasOneOrMany
                         */
                        $subQuery->leftJoin(
                            $relationModel->getRelated()->getTable() . ' AS relatedSort',
                            $relationModel->getParent()->getQualifiedKeyName(),
                            '=',
                            'relatedSort.' . $relationModel->getForeignKeyName()
                        );


                        break;
                    case 'BelongsTo':
                        /**
                         * @var $relationModel BelongsTo
                         */
                        $subQuery->leftJoin(
                            $relationModel->getRelated()->getTable() . ' AS relatedSort',
                            method_exists($relationModel, 'getQualifiedForeignKey')
                                ? $relationModel->getQualifiedForeignKey()
                                : $relationModel->getQualifiedForeignKeyName(),
                            '=',
                            'relatedSort.' . (method_exists($relationModel, 'getOwnerKey')
                                ? $relationModel->getOwnerKey()
                                : $relationModel->getOwnerKeyName())
                        );
                        break;
                    case 'BelongsToMany':
                        /**
                         * @var $relationModel BelongsToMany
                         */
                        $subQuery->leftJoin(
                            $relationModel->getTable(),
                            $relationModel->getQualifiedForeignPivotKeyName(),
                            '=',
                            $model->getModel()->getQualifiedKeyName()
                        )->leftJoin(
                            $relationModel->getRelated()->getTable() . ' AS relatedSort',
                            $relationModel->getQualifiedRelatedPivotKeyName(),
                            '=',
                            'relatedSort.' . $relationModel->getRelated()->getKeyName()
                        );
                        break;

                    default:
                        continue;
                };

                $subQuery->groupBy($model->getModel()->getTable() . '.id');
                $subQuery->addSelect(\DB::raw($aggregate . '( ' . 'relatedSort.' . $param . ' ) AS sortableParam'));

                $subQueryAlias = 'sortQuery-' . crc32(implode('-', [$relation, $param, $sortIndex]));

                $model->joinSub(
                    $subQuery->getQuery(),
                    $subQueryAlias,
                    $model->getModel()->getQualifiedKeyName(),
                    '=',
                    $subQueryAlias . '.id'
                );
                $model->orderBy($subQueryAlias . '.sortableParam', $sort['direction']);
            } else {
                $model->orderBy($sort['key'], $sort['direction']);
            }
        }

        return $model;
    }

    /**
     * @param $model
     * @return mixed
     */
    public function eagerLoadRelations($model)
    {
        $visibleRelations = $this->bagaaravelConfig->getVisibleRelationsForModel($this->modelKey);
        $relations = array_filter($visibleRelations, function ($relation) {
            return method_exists($this->model, $relation);
        });
        if (count($relations)) {
            return $model->with($relations);
        }

        return $model;
    }

    /**
     * @param null $model
     * @return mixed
     */
    public function getAll($model = null)
    {
        if ($model === null) {
            $model = $this->model->select($this->model->getTable() . '.*');
        }

        $model = $this->applyFilters($model);
        $model = $this->applySorting($model);
        $model = $this->eagerLoadRelations($model);

        return $model->get();
    }

    /**
     * @param int  $per_page
     * @param null $model
     * @return mixed
     */
    public function getAllPaginated($per_page = 20, $model = null)
    {
        if ($model === null) {
            $model = $this->model->select($this->model->getTable() . '.*');
        }

        $model = $this->applyFilters($model);
        $model = $this->applySorting($model);
        $model = $this->eagerLoadRelations($model);

        return $model->paginate($per_page);
    }


    /**
     * @param $scope
     * @return mixed
     */
    private function getAllWhereModel($scope)
    {
        $model = $this->eagerLoadRelations($this->model);
        foreach ($scope as $field => $value) {
            if (strpos($value, '%') !== false) {
                $model = $model->where($field, 'LIKE', $value);
            } else {
                $model = $model->where($field, '=', $value);
            }
        }

        return $model;
    }

    /**
     * @param $scope
     * @return mixed
     */
    public function getAllWhere($scope)
    {
        $model = $this->getAllWhereModel($scope);
        $model = $this->eagerLoadRelations($model);

        return $model->get();
    }


    /**
     * @param     $scope
     * @param int $per_page
     * @return mixed
     */
    public function getAllWherePaginated($scope, $per_page = 20)
    {
        $model = $this->getAllWhereModel($scope);
        $model = $this->eagerLoadRelations($model);

        return $model->paginate($per_page);
    }

    /**
     * @param $id
     * @return mixed
     */
    public function getById($id)
    {
        $model = $this->eagerLoadRelations($this->model);

        return $model->findOrFail($id);
    }


    /**
     * @param array $ids
     * @return mixed
     */
    public function getAllIn(array $ids)
    {
        $model = $this->eagerLoadRelations($this->model);

        return $model->whereIn('id', $ids)->get();
    }

    /**
     * @param null $input
     * @return mixed
     */
    public function create($input = null)
    {
        if ($input) {
            return $this->model->create($input);
        } else {
            return $this->model->newInstance();
        }
    }

    /**
     * @param $id
     * @param $input
     * @return bool|mixed
     */
    public function save($id, $input)
    {
        $row = $this->getById($id);
        if ($row) {
            $row->fill($input)->save();

            return $row;
        }

        return false;
    }

    /**
     * @param $id
     * @return mixed
     */
    public function delete($id)
    {
        $row = $this->getById($id);

        return $row->delete();
    }
}
