<?php

namespace Bagaaravel\Api\Requests;

use Bagaaravel\Rules\RequiredRelationship;
use Bagaaravel\Rules\RequiredRelationshipDirect;
use Bagaaravel\Utils\ConfigHandler;
use Illuminate\Contracts\Validation\Validator;

class GenericJsonApiFormRequest extends GenericFormRequest implements FormRequestInterface
{

    protected $failedRules = [];

    protected $relationships = [];

    protected $isRelationshipRequest;

    protected function prepareForValidation()
    {
        $this->relationships = $this->input('data.relationships', []);
        $segments = $this->segments();
        $this->isRelationshipRequest = count($segments) >= 2 && $segments[count($segments) - 2] == 'relationships';
    }

    public function authorize()
    {
        return true;
    }


    public function rules()
    {
        $rules = parent::rules();
        //if it is a relationships request, disable validation rules, leaving only relationship rule
        if ($this->isRelationshipRequest) {
            $rules = [];
        }
        
        if ($this->getMethod() !== 'PATCH') {
            if ($key = $this->getModelName()) {
                $relationships = config("bagaaravel-validation.jsonapi.${key}.relationships", []);
            }

            if ($this->isRelationshipRequest) {
                $field = array_values(array_slice($this->segments(), -1))[0];
                $rules['data'] = [new RequiredRelationshipDirect(
                    config("bagaaravel.relationships_required", true),
                    ConfigHandler::getRelationRequiredStatus($relationships[$field])
                )];
                return $rules;
            }


            foreach ($relationships as $field => $relationship) {
                $rules["relationships.$field"] = [$this->getRelationshipRule($relationship)];
            }
        }

        return $rules;
    }


    /**
     * @param Validator $validator
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        $this->failedRules = $validator->failed();
        if (app()->version() < '5.5') {
            return parent::failedValidation($validator);
        }

        $response_errors = [];
        $errors = $validator->errors()->toArray();

        foreach ($errors as $field => $messages) {

            foreach ($messages as $message_index => $message) {

                $rule = array_keys($this->failedRules[$field])[$message_index];

                $meta = $this->failedRules[$field];
                $meta[$message] = $meta[$rule];
                unset($meta[$rule]);

                $pointer = $this->isRelationshipRequest ? '/' . $field : '/data/attributes/' . $field;
                if ($rule == RequiredRelationship::class) {
                    $pointer = '/data/' . str_replace('.', '/', $field);//switching from dot notation to slash one
                }

                $err = [
                    'status' => 422,
                    'source' => ['pointer' => $pointer],
                    'title' => $rule,
                    'detail' => $message,
                    'meta' => $meta
                ];

                $response_errors[] = $err;
            }
        }

        throw new \Illuminate\Http\Exceptions\HttpResponseException(response()->json(['errors' => $response_errors], 422)->withHeaders(['Content-Type' => 'application/vnd.api+json']));
    }


    /**
     * Get the proper failed validation response for the request.
     *
     * @param  array $errors
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function response(array $errors)
    {
        $response_errors = [];

        foreach ($errors as $field => $messages) {

            foreach ($messages as $message_index => $message) {

                $rule = array_keys($this->failedRules[$field])[$message_index];
                $pointer = '/data/attributes/' . $field;
                if (stristr($field, 'relationships')) {
                    $pointer = '/data/relationships/';
                }

                $err = [
                    'status' => 422,
                    'source' => ['pointer' => $pointer],
                    'title' => $rule,
                    'detail' => $message
                ];

                $response_errors[] = $err;
            }
        }

        return response()->json(['errors' => $response_errors], 422)->withHeaders(['Content-Type' => 'application/vnd.api+json']);
    }

    /**
     * @throws \HttpException
     */
    public function forbiddenResponse()
    {
        abort(403);
    }

    public function checkPermissions($model)
    {
        $instance = $this->getModelInstance($model);

        $all = $this->all();
        if (isset($all['data'])) {
            if (isset($all['data']['attributes'])) {
                $all['data']['attributes'] = $this->filterWritable($instance, $all['data']['attributes']);
                $this->replace($all);
            }
            if (isset($all['data']['relationships'])) {
                $all['data']['relationships'] = $this->filterRelationWritable($instance, $all['data']['relationships']);
                $this->replace($all);
            }
        }
    }


    /**
     * @return array|string
     */
    protected function validationData()
    {
        if ($this->isRelationshipRequest) {
            return $this->input();
        }

        $dataToValidate = $this->input('data.attributes', []);

        if ($this->relationships) {
            $dataToValidate['relationships'] = $this->input('data.relationships', []);
        }
        return $dataToValidate;
    }

    protected function getRelationshipRule($relationship)
    {
        return new RequiredRelationship(
            config("bagaaravel.relationships_required", true),
            ConfigHandler::getRelationRequiredStatus($relationship));
    }

}