<?php

namespace Bagaaravel\Commands;

use DB;
use Doctrine\DBAL\Schema\Column;
use Illuminate\Console\Command;
use Illuminate\Database\QueryException;
use Riimu\Kit\PHPEncoder\PHPEncoder;

class GenerateModel extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'bagaar:model {table} {model?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate a model based on the provided table.';

    /**
     * Create a new command instance.
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @throws \Throwable
     * @return mixed
     */
    public function handle()
    {
        $tableName = $this->argument('table');
        $modelClass = $this->argument('model') ?? ucfirst(camel_case(str_singular($tableName)));

        $this->info('Generating "'.$modelClass.'" from ' . $tableName);

        $table = \DB::table($tableName);

        // Make sure the table exists
        try {
            $table->exists();
        }
        catch(QueryException $e)
        {
            $this->error('The table "'.$tableName.'" does not exist!');
            return;
        }

        $modelPath = app_path('Models/'.$modelClass.'.php');

        // Make sure not to overwrite an existing model accidentally
        if(file_exists($modelPath))
        {
            $continue = $this->ask('The model already exists, overwrite? (yes|no)');

            if(!in_array($continue, ['yes', 'y', 'j', 1]))
            {
                $this->error('Aborted');
                return;
            }
        }

        $this->info(' + retrieving table data');

        // Make sure the models directory exists
        if(!file_exists(app_path('Models'))) mkdir(app_path('Models'));

        $fields = DB::connection()
            ->getDoctrineSchemaManager()
            ->listTableColumns($this->argument('table'));

        $fillables = [];
        $casts = [];
        $dates = [];
        $columns = [];
        $relations = [];

        $this->info(' + parsing table data');

        foreach($fields as $key => $field)
        {
            /**
             * @var $field Column
             */
            $type = $field->getType()->getName();

            switch($type)
            {
                case 'json':
                    $casts[$key] = 'array';
                    break;
                case 'boolean':
                    $casts[$key] = 'boolean';
                    break;
                case 'datetime':
                    $dates[] = $key;
                    break;
            }

            // Don't register column if it auto increments or is a default date
            if(
                $field->getAutoincrement() ||
                in_array($key, ['created_at', 'updated_at'])
            ) continue;

            // If a columns end with _id, it's a relation
            if(ends_with($key, '_id'))
            {
                $relation = str_replace('_id', '', $key);
                $relationClass = ucfirst(camel_case($relation));

                $relations[$relation] = $relationClass;
                continue;
            }

            $fillables[] = $key;
        }

        $namespace = app()->getNamespace();

        // Start transforming arrays to strings
        $varEncoder = new PHPEncoder;

        $casts = count($casts) == 0 ? null : $varEncoder->encode(
            $casts,
            [
                'array.inline' => false,
                'array.indent' => 4,
                'array.base' => 4
            ]
        );
        $dates = count($dates) == 0 ? null : $varEncoder->encode($dates);
        $fillables = $varEncoder->encode(
            $fillables,
            [
                'array.inline' => false,
                'array.indent' => 4,
                'array.base' => 4
            ]
        );

        $this->info(' + Generating model');

        // Build the content from a template
        $modelContent = view('bagaaravel::stubs.model')
            ->with(compact('fillables', 'casts', 'dates', 'relations', 'columns', 'namespace', 'modelClass', 'tableName'))
            ->render();

        // Store the model
        file_put_contents($modelPath, "<?php\n" . $modelContent);

        $this->info('Generated model "'.$modelClass.'" at ' . $modelPath);
        $this->info('Don\'t forget to define your relations before running "php artisan bagaar:setup '.$modelClass.'"');
    }
}
