<?php

namespace Bagaaravel\Api\Responders;

use Bagaaravel\Acl\Permissions\Resolver;
use Bagaaravel\Acl\Permissions\ResolverAwareTrait;
use Bagaaravel\Api\Transformers\GenericJsonApiTransformer;
use Bagaaravel\Models\DummyPolicyModel;
use Bagaaravel\Utils\BagaaravelConfig;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;

class JsonApiResponder extends ApiResponder implements ResponderInterface
{

    use ResolverAwareTrait;

    protected $relationshipsEnabled = true;
    protected $relationshipsResponseEnabled = false;

    protected $createResourceAndRelationshipLinks = true;

    protected $relationshipsAsLinks = false;

    protected $collectionPermissionTemplate = null;
    protected $collectionRelationshipsPermissionTemplate = null;

    /**
     * @var BagaaravelConfig $bagaaravelConfig
     */
    protected $bagaaravelConfig;

    public function __construct($config = null)
    {
        parent::__construct($config);
        $this->setHeader('Content-Type', 'application/vnd.api+json');
        $this->resolver = $this->getResolverForModel($this->config['model']);
        $this->bagaaravelConfig = \App::make(BagaaravelConfig::class);
    }

    /**
     * @param boolean $relationshipsEnabled
     */
    public function setRelationshipsEnabled($relationshipsEnabled)
    {
        $this->relationshipsEnabled = $relationshipsEnabled;
    }

    /**
     * @param boolean $relationshipsResponseEnabled
     */
    public function setRelationshipsResponseEnabled($relationshipsResponseEnabled)
    {
        $this->relationshipsResponseEnabled = $relationshipsResponseEnabled;
    }


    /**
     * @param boolean $createResourceAndRelationshipLinks
     */
    public function setCreateResourceAndRelationshipLinks($createResourceAndRelationshipLinks)
    {
        $this->createResourceAndRelationshipLinks = $createResourceAndRelationshipLinks;
    }

    /**
     * @param boolean $relationshipsAsLinks
     */
    public function setRelationshipsAsLinks($relationshipsAsLinks)
    {
        $this->relationshipsAsLinks = $relationshipsAsLinks;
    }

    /**
     * @return GenericJsonApiTransformer|\Illuminate\Foundation\Application|mixed
     */
    protected function getAutoDetectedTransformer()
    {
        $transformer = ucfirst($this->model) . 'Transformer';

        if (class_exists($this->getAppNamespace() . "Transformers\\${transformer}")) {
            return app($this->getAppNamespace() . "TransFormers\\${transformer}");
        }

        return app(GenericJsonApiTransformer::class);
    }

    /**
     * @return array
     */
    private function createLinks($collection = null)
    {
        $links = [
            'self' => \Request::fullUrl()
        ];
        if ($this->pagingEnabled && $collection instanceof LengthAwarePaginator) {
            $links['previous'] = $collection->previousPageUrl();
            $links['next'] = $collection->nextPageUrl();
            $links['first'] = $collection->url(1);
            $links['last'] = $collection->url($collection->lastPage());
        }

        return $links;
    }

    /**
     * @return array
     */
    private function createMeta($collection)
    {
        $meta = [];
        if ($this->pagingEnabled && $collection instanceof LengthAwarePaginator) {
            $meta['total'] = $collection->total();
            $meta['per_page'] = $collection->perPage();
        }

        return $meta;
    }

    /**
     * @return array
     */
    private function createCollectionMeta($collection)
    {
        $meta = [];
        if ($this->pagingEnabled && $collection instanceof LengthAwarePaginator) {
            $meta['total'] = $collection->total();
            $meta['per_page'] = $collection->perPage();
        }
        if ($this->includePermissions) {
            $meta['permissions'] = [
                'type' => class_basename($this->model),
                'attributes' => $this->createCollectionPermissionsMeta($collection),
                'relationships' => $this->createCollectionRelationshipsPermissionsMeta($collection),
            ];
        }

        return $meta;
    }

    private function createRelationshipMeta($item)
    {
        $meta = null;
        if (isset($item->pivot)) {
            //are there keys that don't end in _id -> add to meta data
            $meta = [];
            foreach ($item->pivot->toArray() as $key => $value) {
                if (substr($key, strlen($key) - 3, 3) !== '_id') {
                    $meta[$key] = $value;
                }
            }
        }
        return $meta;
    }


    private function createPermissionsMeta($collection)
    {
        $data = $this->createPermissionsItem($collection);
        if ($this->collectionPermissionTemplate) {
            $data = array_diff($this->collectionPermissionTemplate, $data);
        }

        return $data;
    }

    private function createCollectionPermissionsMeta($collection)
    {
        if ($this->pagingEnabled && method_exists($collection, 'getCollection')) {
            $collection = $collection->getCollection();
        }


        /** @var $collection Collection */
        if ($initialItem = $collection->first()) {
            $this->collectionPermissionTemplate = $this->getPermissions($initialItem, true);
        }


        return $this->collectionPermissionTemplate;
    }

    private function createCollectionRelationshipsPermissionsMeta($collection)
    {
        if ($this->pagingEnabled && method_exists($collection, 'getCollection')) {
            $collection = $collection->getCollection();
        }

        /** @var $collection Collection */
        if ($initialItem = $collection->first()) {
            $this->collectionRelationshipsPermissionTemplate = $this->getRelationshipsPermissions($initialItem, true);
        }

        return $this->collectionRelationshipsPermissionTemplate;
    }

    private function createPermissionsItem($item)
    {
        return [
            'id' => $item->getKey(),
            'type' => class_basename($item->getModel()),
            'attributes' => $this->getPermissions($item),
            'relationships' => $this->getRelationshipsPermissions($item),
        ];
    }

    private function getPermissions($item, $isGeneric = false)
    {
        $data = [];

        foreach ($this->extractPermissions($item, $isGeneric) as $attribute => $permissions) {
            if ($permissions & Resolver::READ) {
                $data[$attribute] = 'r';
                $data[$attribute] .= $permissions & Resolver::WRITE ? 'w' : '';
            }
        }

        return $data;
    }

    private function getRelationshipsPermissions($item, $isGeneric = false)
    {
        $data = [];

        foreach ($this->extractRelationshipsPermissions($item, $isGeneric) as $relationship => $permissions) {
            if ($this->allowRelationshipShow($relationship) && ($permissions & Resolver::READ)) {
                $data[$relationship] = 'r';
                $data[$relationship] .= $permissions & Resolver::WRITE ? 'w' : '';
            }
        }

        return $data;
    }

    private function extractPermissions($item, $isGeneric = false)
    {
        return $isGeneric ? $this->resolver->getGenericPermissions($item) : $this->resolver->getPermissions($item);
    }

    private function extractRelationshipsPermissions($item, $isGeneric = false)
    {
        return $isGeneric ? $this->resolver->getRelationshipsGenericPermissions($item) : $this->resolver->getRelationshipsPermissions($item);
    }

    /**
     * @param $item
     * @param $cb
     */
    private function foreachRelationship($item, $cb)
    {
        //check if item is polymorphic
        $isPolymorphic = false;
        if (method_exists($item, \Illuminate\Support\Str::camel($this->model) . 'able')) {
            $isPolymorphic = true;
        }

        foreach ($this->relationships as $name => $model) {
            if ( $this->allowRelationshipShow($name) && $this->allowRelationshipReadablePermission($name, $item)) {
                if ($isPolymorphic) {
                    $relation = $item->{\Illuminate\Support\Str::camel($this->model) . 'able'};
                } else {
                    $relation = $item->{$name};
                }
                $cb($name, $relation);
            }
        }
    }

    /**
     * @param $item
     * @return array
     */
    private function createRelationshipData($item)
    {
        $data = [
            'type' => class_basename($item->getModel()),
            'id' => $item->getKey()
        ];
        if ($meta = $this->createRelationshipMeta($item)) {
            $data['meta'] = $meta;
        }

        return $data;
    }


    /**
     * @param $relation
     * @return array|null|void
     */
    private function createRelationship($relation)
    {
        $rdata = null;

        if (is_a($relation, Collection::class)) {
            if (!$relation->count()) {
                return;
            }

            if ($this->relationshipsAsLinks) {
                return true;
            }

            $rdata = $relation->map(function ($item) {
                $data = $this->createRelationshipData($item);
                return $data;
            });

        } else {
            if (!$relation) {
                return;
            }

            if ($this->relationshipsAsLinks) {
                return true;
            }

            $rdata = $this->createRelationshipData($relation);
        }

        return $rdata;
    }

    /**
     * @param $relation
     * @return array
     */
    public function allowRelationshipShow($relation)
    {
        return in_array($relation, $this->bagaaravelConfig->getVisibleRelationsForModel($this->model));
    }

    public function allowRelationshipReadablePermission($relation, $model)
    {
        return (bool)$this->filterRelationReadable($model, [$relation => $relation]);
    }

    public function allowRelationshipWritablePermission($relation, $model)
    {
        return (bool)$this->filterRelationWritable($model, [$relation => $relation]);
    }

    /**
     * @param $item
     * @return null
     */
    private function createRelationShips($item)
    {
        $list = null;

        if (!$this->relationshipsEnabled) {
            return $list;
        }

        $this->foreachRelationship($item, function ($relationName, $relation) use (&$list, $item) {
            $relationshipData = $this->createRelationship($relation);

            if ($relationshipData) {
                if (!$this->relationshipsAsLinks) {
                    $list[$relationName]['data'] = $relationshipData;
                }

                if ($this->createResourceAndRelationshipLinks) {
                    if (\Route::has($this->getBaseRouteName() . '.relation.' . $relationName . '.show')) {
                        $list[$relationName]['links'] = [
                            'self' => route($this->getBaseRouteName() . '.relation.' . $relationName . '.show', ['id' => $item->getKey()]),
                            'related' => route($this->getBaseRouteName() . '.related.' . $relationName . '.show', ['id' => $item->getKey()])
                        ];
                    }
                }
            }
        });

        return $list;
    }

    /**
     * @param $item
     * @return array
     */
    private function createDataItem($item, $isIncluded = false)
    {

        $item = $this->setSparseFields($item);

        if ($this->relationshipsResponseEnabled) {
            $dataItem = $this->createRelationshipData($item);
        } else {
            $dataItem = [
                'id' => $item->getKey(),
                'type' => class_basename($item->getModel()),
                'attributes' => $this->transformer->transform($item)
            ];

            if ($this->createResourceAndRelationshipLinks) {
                $dataItem['links'] = [
                    'self' => route($this->getBaseRouteName() . '.show', ['id' => $item->getKey()])
                ];
            }

            $relationships = $this->createRelationShips($item);
            if (!empty($relationships)) {
                $dataItem['relationships'] = $relationships;
            }

            if ($this->includePermissions) {
                $dataItem['meta'] = [
                    'permissions' => [
                        'attributes' => $this->collectionPermissionTemplate
                            ? array_diff_assoc($this->getPermissions($item), $this->collectionPermissionTemplate)
                            : $this->getPermissions($item),
                        'relationships' => $this->collectionRelationshipsPermissionTemplate
                            ? array_diff_assoc($this->getRelationshipsPermissions($item), $this->collectionRelationshipsPermissionTemplate)
                            : $this->getRelationshipsPermissions($item),
                    ]
                ];
            }
        }

        return $dataItem;
    }

    /**
     * @param $collection
     * @param bool $isIncluded
     * @return array|null
     */
    private function createData($collection, $isIncluded = false)
    {
        if ($this->pagingEnabled && method_exists($collection, 'getCollection')) {
            $collection = $collection->getCollection();
        }

        $data = null;
        if (is_a($collection, Collection::class)) {
            $data = [];
            foreach ($collection as $item) {
                if (!$isIncluded || $this->allowsByPolicy('view', $item)){
                    $data[] = $this->createDataItem($item, $isIncluded);
                }
            }
        } elseif ($collection && (!$isIncluded || $this->allowsByPolicy('view', $collection))) {
            $data = $this->createDataItem($collection, $isIncluded);
        }

        return $data;
    }

    /**
     * @param     $collection
     * @param int $status
     * @return mixed
     */
    public function transformAndRespondCollection($collection, $status = 200)
    {
        $return = [];
        $meta = $this->createCollectionMeta($collection);

        $return['data'] = $this->createData($collection);

        $return['links'] = $this->createLinks($collection);

        if (!empty($meta)) {
            $return['meta'] = $meta;
        }

        $included = $this->createIncluded($collection);
        if (!empty($included)) {
            $return['included'] = $included;
        }

        return $this->respondCollection($return, $status);
    }


    /**
     * @param     $item
     * @param int $status
     * @return mixed
     */
    public function transformAndRespondItem($item, $status = 200)
    {
        $return = [
            'data' => $this->createData($item),
            'links' => $this->createLinks()
        ];
        $meta = $this->createMeta($item);
        if (!empty($meta)) {
            $return['meta'] = $meta;
        }
        $included = $this->createIncluded($item);
        if (!empty($included)) {
            $return['included'] = $included;
        }

        return $this->respondItem($return, $status);
    }

    /**
     * @param       $object
     * @param array $headers
     * @return mixed
     */
    public function transformAndRespondContentCreated($object, $headers = [])
    {
        $return = [
            'data' => $this->createData($object),
            'links' => $this->createLinks()
        ];
        $meta = $this->createMeta($object);
        if (!empty($meta)) {
            $return['meta'] = $meta;
        }

        $headers['Location'] = route($this->getBaseRouteName() . '.show', [$object->getKey()]);

        return $this->respondContentCreated($return, $headers);
    }


    /**
     * @return string
     */
    protected function getBaseRouteName()
    {
        $name = request()->route()->getName();
        $nameA = explode('.', $name);
        array_pop($nameA);

        return implode('.', $nameA);
    }

    protected function createIncluded($collection)
    {
        $included = [];
        $includeString = \Request::get('include', '');
        if ($includeString === '') {
            return $included;
        }
        $collection = is_iterable($collection) ? $collection : [$collection];
        foreach (explode(',', $includeString) as $relation) {
            $this->processIncludedRelations($collection, $relation, $included);
        }

        $unique_keys = [];

        return array_values(array_filter($included, function ($item) use (&$unique_keys) {
            $key = $item['type'] . ':' . $item['id'];
            $exists = in_array($key, $unique_keys);

            if (!$exists) {
                $unique_keys[] = $key;
            }

            return !$exists;
        }));
    }

    protected function processIncludedRelations($collection, $relation_path, &$included)
    {
        $relation_parts = explode('.', $relation_path, 2);

        if (isset($relation_parts[0])) {
            $relation = $relation_parts[0];
            if ($this->allowRelationshipShow($relation)) {
                $responder = null;
                $models = new Collection();
                foreach ($collection as $item) {
                    if (!$this->allowRelationshipReadablePermission($relation, $item)) {
                        continue;
                    }
                    $item = $this->setSparseFields($item);
                    $polyRelation = \Illuminate\Support\Str::camel(class_basename($item)) . 'able';
                    if (method_exists($item, $relation) || method_exists($item, $polyRelation)) {
                        $responder = new JsonApiResponder($this->config['relationships'][$relation]);
                        $responder->setCreateResourceAndRelationshipLinks(false);
                        if ($item->{$relation} instanceof Collection) {
                            $included = array_merge($included, $responder->createData($item->{$relation}, true));
                            $models = $models->merge($item->{$relation});
                        }
                        if ($item->{$relation} instanceof $responder->config['model']) {
                            $includedItem = $responder->createData($item->{$relation}, true);
                            if (!is_null($includedItem)) {
                                $included[] = $includedItem;
                                $models[] = $item->{$relation};
                            }
                        }
                        if ($item->{$polyRelation} instanceof $responder->config['model']) {
                            $includedItem = $responder->createData($item->{$polyRelation}, true);
                            if (!is_null($includedItem)) {
                                $included[] = $includedItem;
                                $models[] = $item->{$polyRelation};
                            }
                        }
                    }
                }

                if ($responder && isset($relation_parts[1])) {
                    $responder->processIncludedRelations($models, $relation_parts[1], $included);
                }
            }

        }
    }

    /**
     * @param $item
     * @return mixed
     */
    protected function setSparseFields($item)
    {
        if ($fieldsets = \Request::get('fields', null)) {
            foreach ($fieldsets as $model => $fields) {
                if (ucfirst($model) === class_basename($item)) {
                    $attributes = array_keys($item->getAttributes());
                    $visible = array_filter($attributes, function ($attribute) use ($fields) {
                        return $attribute == 'id' || in_array($attribute, explode(',', $fields));
                    });
                    $item->makeHidden(array_diff($attributes, $visible));
                }
            }
        }

        return $item;
    }

    protected function allowsByPolicy($action, $model)
    {
        $policyExists = false;
        try {
            if (\Gate::getPolicyFor($model)) {
                $policyExists = true;
            }
        } catch (\InvalidArgumentException $e) {
            $policyExists = false;
        }

        $actionExists = false;
        if ( $policyExists ) {
            $policy = \Gate::getPolicyFor($model);
            if ( method_exists($policy, $action) ) {
                $actionExists = true;
            }
        }

        if ($policyExists && $actionExists) {
            return \Gate::allows($action, $model);
        }

        return true;
    }

}
