<?php

namespace Bagaaravel\Commands;

use DB;
use Doctrine\DBAL\Schema\Column;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\Relation;
use ReflectionMethod;
use Riimu\Kit\PHPEncoder\PHPEncoder;

class SetupFromModel extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'bagaar:setup {model}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate the basics for an endpoint, based on the provided model.';

    /**
     * Create a new command instance.
     *
     * @param DB $db
     */
    public function __construct()
    {
        parent::__construct();
    }

    protected $modelClass;
    protected $nameSpace;

    /**
     * Retrieves all the relation methods from a model instance.
     *
     * @param Model $modelInstance
     *
     * @return mixed
     */
    protected function getRelations($modelInstance)
    {
        $reflectionObject = new \ReflectionObject($modelInstance);

        return collect($reflectionObject->getMethods(\ReflectionMethod::IS_PUBLIC))
            ->filter(function (ReflectionMethod $method) use($modelInstance) {
                return str_contains($method->class, get_class($modelInstance));
            })
            ->filter(function (ReflectionMethod $method) use ($modelInstance) {
                if(count($method->getParameters()) > 0) return false;

                try {
                    return $modelInstance->{$method->name}() instanceof Relation;
                }
                catch (\Exception $e)
                {
                    return false;
                }
            })
            ->mapWithKeys(function (ReflectionMethod $method) use ($modelInstance) {
                /**
                 * @var $relation Relation
                 */
                $relation = $modelInstance->{$method->name}();

                $relatedModel = explode('\\', get_class($relation->getRelated()));

                $optional = false;

                // If a belongsTo relation is nullable, it's optional
                if($relation instanceof BelongsTo) {
                    $relationTableName = $relation->getRelated()->getTable();
                    $relationField = $relation->getForeignKey();

                    $relationColumns = $this->getColumnsFromTable($relationTableName);

                    if(isset($relationColumns[$relationField]) && !$relationColumns[$relationField]->getNotnull()){
                        $optional = true;
                    }
                }

                $identifier = array_pop($relatedModel);

                // Mark the relation's required state
                $identifier .= ($optional === true) ? '|optional' : '|required';

                return [
                    $method->name => $identifier,
                ];
            });
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $this->nameSpace = app()->getNamespace();
        $this->modelClass = $this->argument('model');
        $modelFilePath = app_path('Models/' . $this->modelClass . '.php');

        $this->info('Setting up resources for the "' . $this->modelClass . '" model.');

        // Make sure the model exists
        if (!file_exists($modelFilePath)) {
            $this->error('Model could not be loaded from this location: ' . $modelFilePath);
        }

        $modelNameSpaced = $this->nameSpace . 'Models\\' . $this->modelClass;

        $this->info(' + Retrieving database definition.');

        /**
         * @var $modelInstance Model
         */
        $modelInstance = new $modelNameSpaced;
        $tableName = $modelInstance->getTable();

        $fields = $this->getColumnsFromTable($tableName);

        $columns = [];

        foreach ($fields as $key => $field) {
            /**
             * @var $field Column
             */
            $type = $field->getType()->getName();

            $rules = [];

            switch ($type) {
                case 'boolean':
                    $rules[] = 'boolean';
                    break;
                case 'datetime':
                    $rules[] = 'date';
                    break;
                case 'integer':
                    $rules[] = 'integer';
                    break;
                case 'float':
                    $rules[] = 'numeric';
                    break;
            }

            // Don't register column if it auto increments or is a default date
            if (
                $field->getAutoincrement() ||
                in_array($key, ['created_at', 'updated_at'])
            ) {
                continue;
            }

            // If a columns end with _id, it's a relation
            if (\Illuminate\Support\Str::endsWith($key, '_id')) {
                continue;
            }

            if ($field->getNotnull()) {
                $rules[] = 'required';
            }

            $columns[$key]['rules'] = $rules;
        }

        $relations = $this->getRelations($modelInstance);

        $this->info(' + Register in config.');
        $this->addToConfig($columns, $relations);
        $this->info(' + Create and register policy.');
        $this->createPolicy($relations);
        $this->info(' + Create controller and route.');
        $this->createController();
    }

    protected function getColumnsFromTable($tableName)
    {
        return DB::connection()
            ->getDoctrineSchemaManager()
            ->listTableColumns($tableName);
    }

    protected function addToConfig($columns, $relations)
    {
        $existingConfig = config('bagaaravel');

        if (isset($existingConfig['jsonapi'][$this->modelClass])) {
            $this->error('There\'s already a config entry with the same key (' . $this->modelClass . ')');
            $overwrite = $this->handleAnswer($this->ask('Do you want to overwrite it?'));

            if(!$overwrite) return;
        }

        $rules = [];

        foreach ($columns as $key => $column) {
            if (!isset($column['rules'])) {
                continue;
            }
            $rules[$key] = $column['rules'];
        }

        $previousConfig = isset($existingConfig['jsonapi'][$this->modelClass]) ? $existingConfig['jsonapi'][$this->modelClass] : [];

        // Start to build config
        $existingConfig['jsonapi'][$this->modelClass] = [
            'model' => $this->nameSpace . 'Models\\' . $this->modelClass,
        ];

        if (count($rules) > 0) {
            $existingConfig['jsonapi'][$this->modelClass]['validation_rules'] = $rules;
        }

        if ($relations->count() > 0) {

            foreach($relations as $method => $model)
            {
                // Only add the relation if it hasn't been defined before
                if(!isset($existingConfig['jsonapi'][$this->modelClass]['relationships'][$method])) {
                    $existingConfig['jsonapi'][$this->modelClass]['relationships'][$method] = $model;
                }
                if(!isset($existingConfig['jsonapi'][$model])) continue;

                $remoteRelations = $this->getKeyForModelRelation($model);

                if(count($remoteRelations) == 0) continue;

                foreach($remoteRelations as $key => $class)
                {
                    // Skip if already set
                    if(isset($existingConfig['jsonapi'][$model]['relationships'][$key])) continue;
                    $existingConfig['jsonapi'][$model]['relationships'][$key] = $class;
                }
            }
        }

        $varEncoder = new PHPEncoder;
        $encodedConfig = $varEncoder->encode(
            $existingConfig,
            [
                'string.classes' => collect($existingConfig['jsonapi'])->map(function ($resource) {
                    return str_replace(['::class', '\\App'], ['', 'App'], $resource['model']);
                })
                    ->values()
                    ->toArray(),
            ]
        );

        $configContent = "<?php\n\nreturn " . $encodedConfig . ';';

        file_put_contents(config_path('bagaaravel.php'), $configContent);
    }

    protected function createPolicy($relations)
    {
        if (!file_exists(app_path('Policies'))) {
            mkdir(app_path('Policies'));
        }

        $policyPath = app_path('Policies/' . $this->modelClass . 'Policy.php');
        $overwrite = false;

        if (file_exists($policyPath)) {
            $this->error('A policy named "' . $this->modelClass . 'Policy" already exists!');
            $overwrite = $this->handleAnswer($this->ask('Do you want to overwrite it?'));

            if(!$overwrite) return;
        }

        // Create the policy
        $policyContent = view('bagaaravel::stubs.policy')
            ->with('model', $this->modelClass)
            ->with('relations', $relations)
            ->with('namespace', $this->nameSpace)
            ->with('policies', config('bagaaravel.policies'))
            ->render();


        file_put_contents($policyPath, "<?php\n" . $policyContent);

        $this->info('   > Policy created.');

        // Don't re-register it if we're overwriting
        if($overwrite) return;

        // Register the policy
        $modelClassNameSpaced = '\\' . $this->nameSpace . 'Models\\' . $this->modelClass . '::class';
        $policyNameSpaced = '\\' . $this->nameSpace . 'Policies\\' . $this->modelClass . 'Policy::class';

        $providerFilePath = app_path('Providers/AuthServiceProvider.php');
        $authServiceProvider = file_get_contents($providerFilePath);

        $registeredInProvider = str_replace(
            '    protected $policies = [',
            '    protected $policies = [
        ' . $modelClassNameSpaced . ' => ' . $policyNameSpaced . ',',
            $authServiceProvider
        );

        file_put_contents($providerFilePath, $registeredInProvider);
        $this->info('   > Policy registered.');
    }

    protected function createController()
    {
        $controllerPath = app_path('Http/Controllers/Api');

        if (!file_exists($controllerPath)) {
            mkdir($controllerPath);
        }

        $controllerClassPath = app_path('Http/Controllers/Api') . '/' . $this->modelClass . 'Controller.php';

        $overwrite = false;
        if (file_exists($controllerClassPath)) {
            $this->error('A controller named "' . $this->modelClass . 'Controller" already exists!');
            $overwrite = $this->handleAnswer($this->ask('Do you want to overwrite it?'));

            if(!$overwrite) return;
        }

        $controllerContent = "<?php\n\n" . view('bagaaravel::stubs.controller')
                ->with([
                    'model'     => $this->modelClass,
                    'namespace' => $this->nameSpace,
                ]);

        file_put_contents(
            $controllerClassPath,
            $controllerContent
        );
        $this->info('   > Controller created.');

        // Don't add the route again if we're overwriting
        if($overwrite) return;

        $routesContent = file_get_contents(base_path('routes/api.php'));
        $routesContent .= "\nRoute::jsonapi('" . \Illuminate\Support\Str::snake($this->modelClass) . "', 'Api\\" . $this->modelClass . "Controller');";
        file_put_contents(base_path('routes/api.php'), $routesContent);
        $this->info('   > Route registered.');
    }

    /**
     * @param $answer
     *
     * @return bool
     */
    protected function handleAnswer($answer)
    {
        return in_array($answer, ['y', 'yes', 'ja', 'j', 1]);
    }

    protected $models = [];

    protected function getKeyForModelRelation($model)
    {
        if(!isset($this->models[$model])) {
            $modelNameSpaced = $this->nameSpace . 'Models\\' . $model;
            $modelInstance = new $modelNameSpaced;
            $this->models[$model] = $this->getRelations($modelInstance);
        }

        return $this->models[$model]->filter(function($modelClass, $key) use($model) {
            return $modelClass == $this->modelClass;
        });
    }
}
