<?php

namespace Bagaaravel\Api\Transformers;

use Bagaaravel\Acl\Permissions\ResolverAwareTrait;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;

abstract class Transformer implements TransformerInterface
{
    use ResolverAwareTrait;

    const TRANSFORM_ATTRIBUTES = 'transform_attributes';

    protected $config;
    protected $relationships = [];

    /**
     * @param $config
     */
    public function setConfig($config)
    {
        $this->config = config('bagaaravel.jsonapi.' . $config);
        $this->resolver = $this->getResolverForModel($this->config['model'] ?? null);
        if ( isset($this->config['relationships']) ) $this->relationships = $this->config['relationships'];
    }

    /**
     * @param $items
     * @return \Illuminate\Support\Collection
     */
    public function transform($items)
    {
        if ($items instanceof Collection) {
            return $this->transformItems($items);
        }
        return $this->doTransformItem($items);
    }

    /**
     * @param Collection $items
     * @return \Illuminate\Support\Collection
     */
    public function transformItems(Collection $items)
    {
        return $items->map(function ($item) {
            return $this->doTransformItem($item);
        });
    }

    /**
     * @param $item
     * @return mixed
     */
    public abstract function transformItem($item);

    /**
     * @param $item
     * @return mixed
     */
    protected function doTransformItem($item)
    {
        $transformed = $this->transformItem($item);

        return $this->filterReadable($item, $transformed);
    }

    protected function transformAttributesByConfig(array $data)
    {
        $transformConfig = $this->config[self::TRANSFORM_ATTRIBUTES] ?? null;

        if ($transformConfig) {
            foreach ($transformConfig as $name => $value) {
                if (isset($data[$name])) {
                    [$type, $format] = explode('|', $value, 2);
                    switch ($type) {
                        case 'date':
                            $data[$name] = $data[$name] ? (new Carbon($data[$name]))->format($format) : null;
                            break;
                        default:
                            throw new \InvalidArgumentException(sprintf('Transform type \'%s\' not found.', $type));
                    }
                }
            }
        }

        return $data;
    }
}