<?php

namespace Bagaaravel\Api;

use Bagaaravel\Exceptions\ApiException;
use Cartalyst\Sentinel\Laravel\Facades\Reminder;
use Cartalyst\Sentinel\Laravel\Facades\Sentinel;
use Illuminate\Http\Response as ResponseCodes;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Validator;

trait PasswordReminder
{

    private $user;

    // you can override these in your __construct method
    private $mail_templates = [
        'reset' => 'bagaaravel::emails.password_reset',
    ];

    public function reminder()
    {
        $this->findUserOrFail();

        $reminder = Reminder::create($this->user);

        $mail_data = [
            'email' => urlencode($this->user->email),
            'code'  => $reminder->code,
        ];

        // no queue since we (normally) push it directly to the MTA
        Mail::send($this->mail_templates['reset'], $mail_data, function ($m) {
            $m->from(Config::get('bagaaravel.mailers.from_email'), Config::get('bagaaravel.mailers.from_name'));
            $m->to($this->user->email)->subject(trans('bagaaravel::subjects.password_reset'));
        });

        if (!$reminder) {
            throw new ApiException('REMINDER_NOT_SENT', ResponseCodes::HTTP_UNPROCESSABLE_ENTITY);
        }

        return Response::make(null, ResponseCodes::HTTP_OK);
    }

    public function reset()
    {
        $validator = Validator::make(Request::only('password', 'email', 'code'), [
            'email'    => 'required',
            'code'     => 'required',
            'password' => 'required|min:8',
        ]);
        if ($validator->fails()) {
            throw new ApiException($validator->messages(), ResponseCodes::HTTP_UNPROCESSABLE_ENTITY);
        }

        $this->findUserOrFail();
        $reminder = Reminder::complete(
            $this->user,
            Request::input('code'),
            Request::input('password')
        );

        if (!$reminder) {
            throw new ApiException('INVALID_CODE', ResponseCodes::HTTP_UNPROCESSABLE_ENTITY);
        }

        return Response::make(null, ResponseCodes::HTTP_OK);
    }

    private function findUserOrFail()
    {
        $this->user = Sentinel::findByCredentials(['email' => Request::input('email')]);
        if (!$this->user) {
            throw new ApiException('EMAIL_UNKNOWN', ResponseCodes::HTTP_NOT_FOUND);
        }
    }

}
