<?php

namespace Bagaaravel\Api\Repositories;


use Bagaaravel\Api\Exceptions\NoFilterColumnsDefined;
use Illuminate\Database\Eloquent\Model;

trait FilterByImportanceTrait
{

    /**
     * Filter results for the specific user.
     * all other defined filters will be 'OR'.
     *
     * @param Model $model
     *
     * @return mixed
     * @throws NoFilterColumnsDefined
     */
    public function applyFilters($model)
    {
        $this->checkForColumnsDefinition();

        // Primary fields will be 'AND' relations
        foreach ($this->primaryFilters as $key)
        {
            if(isset($this->filters[$key]))
            {
                $model->where('user_id', $this->filters[$key]);
                unset($this->filters[$key]);
            }
        }

        // Any other field is less important
        $model->where(function($q){
            foreach ($this->filters as $field => $value) {
                // make sure the values are in array format
                if (is_array($value)) {
                    $values = $value;
                } elseif (strpos($value, ',') !== false) {
                    $values = explode(',', $value);
                } else {
                    $values = [$value];
                }

                $q->orWhere(function($query) use($field, $values){
                    return $this->addValues($query, $field, $values);
                });
            }

            return $q;
        });

        return $model;
    }

    protected function addValues(&$q, $field, $values)
    {
        foreach ($values as $index => $value) {
            $comparison = strpos($value, '%') !== false ? 'LIKE' : '=';
            if ($index === 0) {
                $q->where($field, $comparison, $value);
            } else {
                $q->orWhere($field, $comparison, $value);
            }
        }

        return $q;
    }

    /**
     * [!!] Make sure a columns property is present on the repository.
     * Within this property you should define 'primary' filters.
     *
     * Problem
     *  you want to filter specific for a user
     *
     * Solution
     * specify user_id within this column,
     * all other fields will become secondary
     *
     * @throws NoFilterColumnsDefined
     */
    protected function checkForColumnsDefinition()
    {
        if(!isset($this->primaryFilters))
        {
            throw new NoFilterColumnsDefined(__CLASS__ . ' has no primary filter(s) defined');
        }
    }
}