<?php

namespace Bagaaravel\Api\Repositories;

use Carbon\Carbon;

trait FilterTrait
{
    protected $filters = [];

    /**
     * @param $filters
     */
    public function setFilters($filters)
    {
        $this->filters = $filters;
    }

    /**
     * @param $model
     * @return mixed
     * @throws \Exception
     */
    public function applyFilters($model)
    {
        $tableName = $model->getModel()->getTable();
        $this->addAlias($tableName);
        $casts = $this->getAvailableCastsFields($model, $tableName);
        $dates = $this->getAvailableDateFields($model, $tableName);

        $relationshipAttributeConditions = [];
        foreach ($this->filters as $field => $value) {
            if (strpos($field ?? '', '.') !== false && strpos($field ?? '', $tableName) !== 0) {
                [$relationName] = explode('.', $field, 2);
                $relationshipAttributeConditions[$relationName][$field] = $value;
            } elseif (in_array($field, array_keys($casts))) {
                $this->applyCastsFieldFilter($casts[$field], $model, $field);
            } elseif (in_array($field, $dates)) {
                $this->applyDateFilter($model, $field);
            } else {
                $this->applyDefaultFieldFilter($model, $field);
            }
        }

        foreach ($relationshipAttributeConditions as $relationName => $relationFilters) {
            $this->applyRelationshipAttributeFilters($model, $relationName, $relationFilters);
        }

        return $model;
    }

    /**
     * @param $model
     * @param $relationName
     * @param $filters
     * @throws \Exception
     */
    protected function applyRelationshipAttributeFilters($model, $relationName, $filters)
    {
        $subQuery = $this->getBuiltJoinQueryWithRelationModel($model, $relationName);
        $relationModel = $model->getModel()->{$relationName}();
        $casts = $this->getAvailableCastsFields($relationModel, $relationName);
        $dates = $this->getAvailableDateFields($relationModel, $relationName);

        foreach ($filters as $field => $value) {
            if (in_array($field, array_keys($casts))) {
                $this->applyCastsFieldFilter($casts[$field], $subQuery, $field);
            } elseif (in_array($field, $dates)) {
                $this->applyDateFilter($subQuery, $field);
            } else {
                $this->applyDefaultFieldFilter($subQuery, $field);
            }
        }

        $model->where(function ($query) use ($subQuery) {
            $query->whereRaw($query->getModel()->getTable() . '.' . $query->getModel()->getKeyName() . ' IN (' . $subQuery->toSql() . ')');
            $query->mergeBindings($subQuery->getQuery());
        });
    }

    /**
     * @param $model
     * @param $filter
     */
    protected function applyDefaultFieldFilter($model, $filter)
    {
        $values = $this->getFilterComparisonValues($this->filters[$filter]);
        $model->where(function ($query) use ($values, $model, $filter) {
            foreach ($values as $index => $value) {
                $comparison = strpos($value ?? '', '%') !== false ? 'LIKE' : '=';
                $where = $index === 0 ? 'where' : 'orWhere';
                $query->$where($filter, $comparison, $value);
            }
        });
    }

    /**
     * @param $castType
     * @param $model
     * @param $filter
     */
    protected function applyCastsFieldFilter($castType, $model, $filter)
    {
        switch ($castType) {
            case 'date':
                $this->applyDateFilter($model, $filter);
                break;
            case 'timestamp':
                $this->applyTimestampFilter($model, $filter);
                break;
            case 'boolean':
                $this->applyBooleanFilter($model, $filter);
                break;
            default:
                throw new \InvalidArgumentException(sprintf('Incorrect type \'%s\' for $casts field \'%s\'.', $castType, $filter));
        }
    }

    /**
     * @param $model
     * @param $filter
     */
    protected function applyDateFilter($model, $filter)
    {
        $values = $this->getFilterComparisonValues($this->filters[$filter]);
        $model->where(function ($query) use ($values, $model, $filter) {
            foreach ($values as $index => $value) {
                $where = $index === 0 ? 'whereDate' : 'orWhereDate';
                $query->$where($filter, $value);
            }
        });
    }

    /**
     * @param $model
     * @param $filter
     */
    protected function applyTimestampFilter($model, $filter)
    {
        $values = $this->getFilterComparisonValues($this->filters[$filter]);
        $model->where(function ($query) use ($values, $model, $filter) {
            foreach ($values as $index => $value) {
                $value = (new Carbon())->setTimestamp($value);
                $where = $index === 0 ? 'where' : 'orWhere';
                $query->$where($filter, $value);
            }
        });
    }

    /**
     * @param $model
     * @param $filter
     */
    protected function applyBooleanFilter($model, $filter)
    {
        $values = $this->getFilterComparisonValues($this->filters[$filter]);
        $model->where(function ($query) use ($values, $model, $filter) {
            foreach ($values as $index => $value) {
                $value = filter_var($value, FILTER_VALIDATE_BOOLEAN);
                $where = $index === 0 ? 'where' : 'orWhere';
                $query->$where($filter, $value);
            }
        });
    }

    private function getBuiltJoinQueryWithRelationModel($model, $relationName)
    {
        $relationModel = $model->getModel()->{$relationName}();
        $relationshipType = class_basename($relationModel);
        $subQuery = app(get_class($model->getModel()))
            ->withoutGlobalScopes()
            ->distinct()
            ->select($model->getModel()->getQualifiedKeyName() . ' AS id')
            ->from($model->getModel()->getTable());

        switch ($relationshipType) {
            case 'HasMany':
            case 'HasOne':
                /**
                 * @var $relationModel HasOneOrMany
                 */
                $subQuery->join(
                    $relationModel->getRelated()->getTable() . ' AS ' . $relationName,
                    $relationModel->getParent()->getQualifiedKeyName(),
                    '=',
                    $relationName . '.' . $relationModel->getForeignKeyName()
                );
                break;
            case 'BelongsTo':
                /**
                 * @var $relationModel BelongsTo
                 */
                $subQuery->join(
                    $relationModel->getRelated()->getTable() . ' AS ' . $relationName,
                    method_exists($relationModel, 'getQualifiedForeignKey')
                        ? $relationModel->getQualifiedForeignKey()
                        : $relationModel->getQualifiedForeignKeyName(),
                    '=',
                    $relationName . '.' . (method_exists($relationModel, 'getOwnerKey')
                        ? $relationModel->getOwnerKey()
                        : $relationModel->getOwnerKeyName())
                );
                break;
            case 'BelongsToMany':
                /**
                 * @var $relationModel BelongsToMany
                 */
                $subQuery->join(
                    $relationModel->getTable(),
                    $relationModel->getQualifiedForeignPivotKeyName(),
                    '=',
                    $model->getModel()->getQualifiedKeyName()
                )->join(
                    $relationModel->getRelated()->getTable() . ' AS ' . $relationName,
                    $relationModel->getQualifiedRelatedPivotKeyName(),
                    '=',
                    $relationName . '.' . $relationModel->getRelated()->getKeyName()
                );
                break;
            default:
                break;
        };

        return $subQuery;
    }

    /**
     * @param $value
     * @return array
     */
    private function getFilterComparisonValues($value): array
    {
        if (is_array($value)) {
            $values = $value;
        } elseif (strpos($value ?? '', ',') !== false) {
            $values = explode(',', $value);
        } else {
            $values = [$value];
        }

        return $values;
    }

    private function getAvailableCastsFields($model, $alias): array
    {
        $casts = [];
        foreach ($model->getModel()->getCasts() as $name => $type) {
            if (in_array($type, ['date', 'timestamp', 'boolean'])) {
                $casts[$alias . '.' . $name] = $type;
            }
        }
        return $casts;
    }

    private function getAvailableDateFields($model, $alias): array
    {
        $fields = [];
        foreach ($model->getModel()->getDates() as $field) {
            $fields[] = $alias . '.' . $field;
        }
        return $fields;
    }

    private function addAlias($alias)
    {
        foreach ($this->filters as $name => $value) {
            $newFilterName = sprintf('%s.%s', $alias, $name);
            if (strpos($name ?? '', '.') === false) {
                $this->filters[$newFilterName] = $value;
                unset($this->filters[$name]);
            }
        }
    }
}
