<?php

namespace Bagaaravel\Http\Controllers\Auth;

use Bagaaravel\Http\Controllers\Controller;
use Bagaaravel\Http\Requests\PasswordResetRequest;
use Bagaaravel\Api\Responders\JsonApiResponder;
use Illuminate\Foundation\Auth\ResetsPasswords;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Password;

class ResetPasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset requests
    | and uses a simple trait to include this behavior. You're free to
    | explore this trait and override any methods you wish to tweak.
    |
    */

    use ResetsPasswords;

    /**
     * Where to redirect users after resetting their password.
     *
     * @var string
     */
    protected $redirectTo = '/home';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }
    /**
     * Get the password reset credentials from the request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array
     */
    protected function credentials(Request $request)
    {
        return $request->only(
            'data.attributes.email',
            'data.attributes.password',
            'data.attributes.password_confirmation',
            'data.attributes.token'
        );
    }

    /**
     * Reset the given user's password.
     *
     * @param  PasswordResetRequest $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function reset(PasswordResetRequest $request)
    {

        $response = $this->broker()->reset(
            $this->credentials($request)['data']['attributes'], function ($user, $password) {
            $this->resetPassword($user, $password);
        }
        );

        return $response == Password::PASSWORD_RESET
            ? $this->sendResetResponse($request, $response)
            : $this->sendResetFailedResponse($request, $response);
    }


    /**
     * Get the response for a successful password reset.
     *
     * @param  string $response
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    protected function sendResetResponse($response)
    {
        return response()->json(null, 201)->withHeaders(['Content-Type' => 'application/vnd.api+json']);
    }

    /**
     * Get the response for a failed password reset.
     *
     * @param  PasswordResetRequest $request
     * @param  string                   $response
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    protected function sendResetFailedResponse(PasswordResetRequest $request, $response)
    {
        return response()->json([
            'errors' => [
                [
                    'source' => ['pointer' => '/data/attributes/token'],
                    'status' => 422,
                    'detail' => trans($response)
                ]
            ]
        ], 422)->withHeaders(['Content-Type' => 'application/vnd.api+json']);
    }
}
