<?php

namespace Bagaaravel\Laravel;

use Bagaaravel\Api\Requests\GenericJsonApiFormRequest;
use Bagaaravel\Api\Responders\ApiResponder;
use Bagaaravel\Api\Responders\JsonApiResponder;
use Bagaaravel\Api\Responders\ResponderInterface;
use Bagaaravel\Commands\GenerateModel;
use Bagaaravel\Commands\SetupFromModel;
use Bagaaravel\Laravel\Router\RefreshableRouter;
use Bagaaravel\Models\DummyPolicyModel;
use Bagaaravel\Policies\DummyPolicy;
use Bagaaravel\Utils\BagaaravelConfig;
use Bagaaravel\Utils\ConfigHandler;
use Illuminate\Routing\Router;
use Illuminate\Support\ServiceProvider;
use Bagaaravel\Api\Requests\GenericFormRequest;
use Bagaaravel\Api\Requests\FormRequestInterface;
use Bagaaravel\Acl\Models\Permission;

class BagaaravelServiceProvider extends ServiceProvider
{
    protected $controller;

    protected $policies = [
        DummyPolicyModel::class => DummyPolicy::class
    ];

    public function boot()
    {
        if ($this->app->runningInConsole()) {
            $this->commands([
                \Bagaaravel\Commands\BagaaravelAclDefaults::class
            ]);
        } else {
            $this->registerPermissions();
        }

        $this->app->register('Bagaaravel\Laravel\BagaaravelRouteServiceProvider');

        $this->registerPolicies();

        $this->loadTranslationsFrom(__DIR__ . '/../translations', 'bagaaravel');
        $this->loadViewsFrom(__DIR__ . '/../views', 'bagaaravel');

        $appNamespace = $this->getAppNamespace();

        Router::macro('jsonapi', function($name, $controller, array $options = []) use ($appNamespace){
            \Route::resource($name, $controller, array_merge([
                'only' => ['index', 'store', 'update', 'show', 'destroy'],
                'parameters' => [$name => 'id'],
            ], $options));

            $groupStack = \Route::getGroupStack();
            $lastGroup = array_pop($groupStack);

            $full_controller_path = '\\' . $lastGroup['namespace'] . '\\' . $controller;
            $config_key = $full_controller_path::$model;
            $config = config('bagaaravel.jsonapi.' . $config_key);

            if ( isset($config['relationships']) ){
                foreach( $config['relationships'] as $rel_name => $rel_config_key ){
                    \route::get($name . '/{id}/' . $rel_name, ['as' => $name . '.related.' . $rel_name . '.show', 'uses' => $controller . '@related_resource_show' ]);
                    \route::get($name . '/{id}/relationships/' . $rel_name, ['as' => $name . '.relation.' . $rel_name . '.show', 'uses' => $controller . '@relationships_show' ]);
                    \route::post($name . '/{id}/relationships/' . $rel_name, ['as' => $name . '.relation.' . $rel_name . '.update', 'uses' => $controller . '@relationships_store' ]);
                    \route::delete($name . '/{id}/relationships/' . $rel_name, ['as' => $name . '.relation.' . $rel_name . '.update', 'uses' => $controller . '@relationships_delete' ]);
                    \route::match(['PUT', 'PATCH'], $name . '/{id}/relationships/' . $rel_name, ['as' => $name . '.relation.' . $rel_name . '.update', 'uses' => $controller . '@relationships_update' ]);

//                    \route::post($name . '/{id}/' . $rel_name, ['as' => $name . '.relation.' . $rel_name . '.store', 'uses' => $controller . '@relation_store_'.$rel_name ]);
//                    \route::match(['PUT', 'PATCH'], $name . '/{id}/' . $rel_name, ['as' => $name . '.relation.' . $rel_name . '.update', 'uses' => $controller . '@relation_update_'.$rel_name ]);
//                    \route::delete($name . '/{id}/' . $rel_name, ['as' => $name . '.relation.' . $rel_name . '.delete', 'uses' => $controller . '@relation_delete_'.$rel_name ]);
                }
            }
        });
    }

    public function register()
    {
        $this->prepareResources();

        $this->app->singleton(BagaaravelConfig::class, function() {
            return new BagaaravelConfig(config('bagaaravel'));
        });
        $this->commands([
            GenerateModel::class,
            SetupFromModel::class,
        ]);


        $this->app->bind(FormRequestInterface::class, function () {
            $route = request()->route();

            if($route !== null) {
                list($controller, $action) = explode('@', $route->getActionName());
                $key = $controller::$model;
                $formRequest = ucfirst($key) . ucfirst($action) . 'Request';

                if (class_exists($this->getAppNamespace() . "Http\\Requests\\${formRequest}")) {
                    $request = app($this->getAppNamespace() . "Http\\Requests\\${formRequest}");
                } else {
                    $controller_parents = class_parents($controller);
                    if (in_array('Bagaaravel\Api\JsonApiController', $controller_parents)) {
                        $request = app(GenericJsonApiFormRequest::class);
                    } else {
                        $request = app(GenericFormRequest::class);
                    }
                }
            }
            else
            {
                $request = app(GenericFormRequest::class);
            }

            return $request;
        });

        $this->app->bind(ResponderInterface::class, function($app){

            if( \App::runningInConsole() && $this->app->environment() != 'behat' ){
                return new JsonApiResponder( (isset($params[0]) ? $params[0] : null) );
            }

            list($controller, $action) = explode('@', request()->route()->getActionName());

            $constructorParam = $controller::$model;

            $controller_parents = class_parents($controller);
            if ( in_array('Bagaaravel\Api\JsonApiController', $controller_parents) ){
                return new JsonApiResponder($constructorParam);
            }

            return new ApiResponder($constructorParam);
        });

        if( \App::runningInConsole() ) {
            $this->app->singleton('router', function ($app) {
                return new RefreshableRouter($app['events'], $app);
            });
        }
    }

    /**
     * Prepare the package resources.
     *
     * @return void
     */
    protected function prepareResources()
    {
        // Publish config
        $this->mergeConfigFrom(__DIR__ . '/../config/bagaaravel.php', 'bagaaravel');
        $this->publishes([__DIR__ . '/../config/bagaaravel.php' => config_path('bagaaravel.php')]);

        $this->mergeConfigFrom(__DIR__ . '/../config/bagaaravel-acl.php', 'bagaaravel-acl');
        $this->publishes([__DIR__ . '/../config/bagaaravel-acl.php' => config_path('bagaaravel-acl.php')]);

        // Publish migrations
        $this->loadMigrationsFrom(__DIR__ . '/../migrations');

        // publish translations
        $this->publishes([__DIR__ . '/../translations' => base_path('resources/lang/vendor/bagaaravel')]);

        // publish views
        $this->publishes([__DIR__ . '/../views' => base_path('resources/views/vendor/bagaaravel')]);

        $this->replaceValidationRulesInConfig();
    }


    public function registerPolicies()
    {
        foreach ($this->policies as $key => $value) {
            \Gate::policy($key, $value);
        }
    }

    public function registerPermissions()
    {
        foreach ($this->getPermissions() as $permission) {
            \Gate::define($permission->slug, function ($user) use ($permission) {
                return $user->hasPermission($permission);
            });
        }
    }

    protected function getAppNamespace()
    {
        return \Illuminate\Container\Container::getInstance()->getNamespace();
    }

    protected function getPermissions()
    {
        return Permission::with('roles')->get();
    }

    protected function replaceValidationRulesInConfig()
    {
        $replacementConfig = [];
        $initialConfig['bagaaravel-validation'] = config('bagaaravel');
        $replacementConfig['bagaaravel'] = $initialConfig['bagaaravel-validation'];
        if (isset($replacementConfig['bagaaravel']['jsonapi'])) {
            array_walk($replacementConfig['bagaaravel']['jsonapi'],
                function (&$model) {
                    if(isset($model['relationships'])){
                        array_walk($model['relationships'], function(&$relationship) use ($model){
                            $relationship = ConfigHandler::extractRelationType($relationship);
                        });
                    }
                });
        }

        config($replacementConfig);
        config($initialConfig);

    }
}
