<?php

namespace Bagaaravel\Acl\Permissions;

class Resolver implements ResolverInterface
{
    const READ = 1;

    const WRITE = 2;

    const ALLOW = '+';

    const DENY = '-';

    const PERMISSION_MAP = [
        'r' => self::READ,
        'w' => self::WRITE,
    ];

    const UPDATE_ACTION = 'update';

    /**
     * @var BasePermission
     */
    protected $permission;

    protected $compiled = [];

    protected $compiledGeneric = [];

    protected $action;

    public function __construct($config, $action = null)
    {
        if (is_array($config)) {
            $this->permission = new BasePermission($config);
        } else {
            $this->permission = app($config);
        }
        $this->action = $action ?? self::UPDATE_ACTION;
    }

    public function isReadable($model, $attribute)
    {
        return $this->getPermission($model, $attribute) & static::READ;
    }

    public function isWritable($model, $attribute)
    {
        return $this->getPermission($model, $attribute) & static::WRITE;
    }

    public function getPermissions($model)
    {
        $permissions = [];

        foreach ($this->permission->getAttributes() as $attribute) {
            $permissions[$attribute] = $this->getPermission($model, $attribute);
        }

        return $permissions;
    }

    public function getRelationshipsPermissions($model)
    {
        $permissions = [];

        foreach ($this->permission->getRelationships() as $relationship) {
            $permissions[$relationship] = $this->getPermission($model, $relationship);
        }

        return $permissions;
    }

    /**
     * Provides common set of permissions on model without narrowing to the exact user or model
     *
     * @return array
     */
    public function getGenericPermissions()
    {
        $permissions = [];

        foreach ($this->permission->getAttributes() as $attribute) {
            $permissions[$attribute] = $this->getGenericPermission($attribute);
        }

        return $permissions;
    }

    public function getRelationshipsGenericPermissions()
    {
        $permissions = [];

        foreach ($this->permission->getRelationships() as $relationship) {
            $permissions[$relationship] = $this->getGenericPermission($relationship);
        }

        return $permissions;
    }

    protected function getPermission($model, $attribute)
    {
        if (!isset($this->compiled[$attribute])) {
            $this->compilePermission($attribute);
        }

        return is_callable($this->compiled[$attribute]) ? $this->compiled[$attribute]($model) : $this->compiled[$attribute];
    }

    protected function getGenericPermission($attribute)
    {
        if (!isset($this->compiledGeneric[$attribute])) {
            $this->compileGenericPermission($attribute);
        }

        return $this->compiledGeneric[$attribute];
    }

    protected function compilePermission($attribute)
    {
        // get defaults
        $permissions = $this->mergeRules(null, $this->permission->getRule($attribute));

        if (auth()->check()) {
            //get permissions by roles
            foreach (auth()->user()->roles->pluck('slug') as $role) {
                $permissions = $this->mergeRules($permissions, $this->permission->getRule($attribute, $role));
            }
            $this->compiled[$attribute] = $this->permissionChecker($attribute, $permissions);

        } else {
            $this->compiled[$attribute] = $permissions;
        }
    }

    protected function permissionChecker($attribute, $permissions)
    {
        $attributeMethodName = 'rightsFor' . ucfirst($attribute ?? '');
        $relationshipMethodName = 'relationshipRightsFor' . ucfirst($attribute ?? '');

        $methodName = null;
        if (method_exists($this->permission, $attributeMethodName)) {
            $methodName = $attributeMethodName;
        } elseif (method_exists($this->permission, $relationshipMethodName)) {
            $methodName = $relationshipMethodName;
        }

        //checking only defined policies
        $policyBasedRule = function ($model) {
            if(\Gate::getPolicyFor($model) && method_exists(\Gate::getPolicyFor($model), $this->action)) {
                return \Gate::forUser(auth()->user())->allows($this->action, $model) ?  null : '-w';
            } else {
                return null;
            }
        };

        if (!is_null($methodName)) {
            return function ($model) use ($methodName, $policyBasedRule, $permissions, $relationshipMethodName) {
                $rule = $this->permission->{$methodName}(auth()->user(), $model);
                return $this->mergeRules($this->mergeRules($permissions, $rule), $policyBasedRule($model));
            };

        } else {
            return function ($model) use ($policyBasedRule, $permissions) {
                return $this->mergeRules($permissions, $policyBasedRule($model));
            };
        }
    }


    protected function compileGenericPermission($attribute)
    {
        // get defaults
        $permissions = $this->mergeRules(null, $this->permission->getRule($attribute));

        if (auth()->check()) {
            //get permissions by roles
            foreach (auth()->user()->roles->pluck('slug') as $role) {
                $permissions = $this->mergeRules($permissions, $this->permission->getRule($attribute, $role));
            }
            $attributeMethodName = 'rightsFor' . ucfirst($attribute ?? '');
            $relationMethodName = 'relationshipRightsFor' . ucfirst($attribute ?? '');
            //skipping context dependent rules
            if (method_exists($this->permission, $attributeMethodName) || method_exists($this->permission, $relationMethodName)) {
                $this->compiledGeneric[$attribute] = 0;
            } else {
                $this->compiledGeneric[$attribute] = $permissions;
            }
        } else {
            $this->compiledGeneric[$attribute] = $permissions;
        }
    }

    protected function mergeRules($initial = 0, $rule = '')
    {
        if ( $rule === null ) $rule = '';

        $merged = $initial;
        foreach (static::PERMISSION_MAP as $token => $permissionMask) {
            $permissionPosition = strpos($rule, $token);
            $permissionModifierPosition = $permissionPosition ? $permissionPosition - 1 : false;
            $permissionModifier = $permissionModifierPosition !== false
                ?
                (
                in_array($rule[$permissionModifierPosition], [static::DENY, static::ALLOW])
                    ? $rule[$permissionModifierPosition] : static::ALLOW
                )
                : static::ALLOW;

            $ruleMask = array_reduce(array_values(static::PERMISSION_MAP), function ($set1, $set2) {
                return $set1 | $set2;
            });

            if ($permissionModifier == static::DENY) {
                $ruleMask = $ruleMask ^ $permissionMask;
                $merged = $merged & $ruleMask;
            } else {
                $merged = $merged | ($permissionPosition !== false ? $permissionMask : 0);
            }
        }
        return $merged;
    }
}
