<?php
namespace Bagaar\BehatMinkApi;

use Behat\Behat\Tester\Exception\PendingException;
use Behat\Behat\Context\Context;
use Behat\Behat\Context\SnippetAcceptingContext;
use Behat\Gherkin\Node\PyStringNode;
use Behat\Gherkin\Node\TableNode;
use Behat\MinkExtension\Context\MinkContext;
use PHPUnit_Framework_Assert;


/**
 * Defines application features from the specific context.
 */
class ApiBaseFeatureContext extends MinkContext implements Context, SnippetAcceptingContext
{
    use \Laracasts\Behat\Context\Migrator;
    use \Laracasts\Behat\Context\DatabaseTransactions;

    /**
     * @var \Symfony\Component\HttpKernel\Client
     */
    protected $client;

    /**
     * @var Symfony\Component\HttpFoundation\Response
     */
    protected $response;

    protected $placeholders = [];

    protected $oauth_access_token;
    protected $add_header_bearer = false;

    protected $json_response;

    protected $form_fields = [];
    protected $json_api_attributes = [];


    private function validateUri($uri){
        if ( $uri[0] != '/' ) $uri = '/' . $uri;
        foreach( $this->placeholders as $ph ){
            $uri = str_replace($ph, $this->getPlaceholder($ph), $uri);
        }
        return $uri;
    }

    private function parseJsonResponse()
    {
        $this->json_response = json_decode($this->response->getContent());
    }

    private function parseKey($key)
    {
        return explode('.', $key);
    }

    private function addFormField($key, $value)
    {
        $this->form_fields[$key] = $value;
    }

    private function addJsonApiAttribute($key, $value)
    {
        $this->json_api_attributes[$key] = $value;
    }

    /**
     * @return \Symfony\Component\HttpKernel\Client
     */
    protected function getClient()
    {
        $this->client = $this->getSession()->getDriver()->getClient();
        return $this->client;
    }

    /**
     * @param $method
     * @param $uri
     * @param array $parameters
     * @param array $files
     * @param array $server
     * @param null $content
     * @param bool|true $changeHistory
     */
    protected function sendRequest($method, $uri, $parameters = [], $files = [], $server = [], $content = null, $changeHistory = true)
    {
        $this->getClient();

        $uri = $this->validateUri($uri);

        if ( $this->add_header_bearer && $this->oauth_access_token ){
            $server['HTTP_Authorization'] = 'Bearer ' . $this->oauth_access_token;
        }

        if ( ! $content && count( $this->form_fields ) ){
            $content = http_build_query( $this->form_fields, null, '&' );
            $parameters = $this->form_fields;

            $this->form_fields = [];

            if ( ! isset($server['CONTENT_TYPE']) ) $server['CONTENT_TYPE'] = 'application/x-www-form-urlencoded';
        }

        if ( ! $content && count( $this->json_api_attributes ) ){
            $json = [
                'data' => [
                    'attributes' => $this->json_api_attributes
                ]
            ];
            $content = json_encode($json);

            $this->json_api_attributes = [];

            if ( ! isset($server['CONTENT_TYPE']) ) $server['CONTENT_TYPE'] = 'application/vnd.api+json';
        }

        $this->client->request($method, $uri, $parameters, $files, $server, $content, $changeHistory);

        $this->response = $this->client->getResponse();
    }


    /**
     * @param $username
     * @param $password
     */
    protected function oauthAuthentication($username, $password)
    {
        $request = [
            'username' => $username,
            'password' => 'password',
            'grant_type' => 'password'
        ];

        $this->sendRequest('POST', 'oauth/token', [], [], ['CONTENT_TYPE' => 'application/json'], json_encode($request));

        $this->theResponseCodeShouldBe(200);
        $this->theJsonResponseShouldHaveAKey('access_token');
        $this->theJsonResponseShouldHaveAKey('token_type');
        $this->theJsonResponseShouldHaveAKey('expires_in');
        $this->theJsonResponseShouldHaveAKey('refresh_token');

        $this->parseJsonResponse();

        $this->oauth_access_token = $this->json_response->access_token;
    }

    protected function getPlaceholder( $placeholder )
    {
        //overrule this method in your class
    }


    /**
     * @Then /^I authenticate with username "([^"]*)" and password "([^"]*)"$/
     */
    public function iAuthenticateWithUsernameAndPassword($username, $password)
    {
        $this->oauthAuthentication($username, $password);
    }


    /**
     * @When /^I send a "([^"]*)" request to "([^"]*)"$/
     *
     * @param $method
     * @param $uri
     */
    public function iSendARequestTo($method, $uri)
    {
        $this->sendRequest($method, $uri);
    }

    /**
     * @When /^I send a "([^"]*)" request with bearer to "([^"]*)"$/
     */
    public function iSendARequestWithBearerTo($method, $uri)
    {
        $this->add_header_bearer = true;
        $this->sendRequest($method, $uri);
        $this->add_header_bearer = false;
    }

    /**
     * @When /^I send a "([^"]*)" json api request with bearer to "([^"]*)"$/
     */
    public function iSendAJsonApiRequestWithBearerTo($method, $uri)
    {
        $this->iSendARequestWithBearerTo($method, $uri);

        $this->parseJsonResponse();

        $this->theResponseShouldBeValidJsonApi();
    }

    /**
     * @When /^I send a "([^"]*)" json api request to "([^"]*)" with attributes$/
     */
    public function iSendAJsonApiRequestToWithAttributes($method, $uri, TableNode $table)
    {
        foreach( $table->getTable() as $row ){
            $this->addJsonApiAttribute($row[0], $row[1]);
        }

        $this->iSendARequestWithBearerTo($method, $uri);
    }


    /**
     * @When /^I send a "([^"]*)" json api request with bearer to "([^"]*)" with attributes$/
     */
    public function iSendAJsonApiRequestWithBearerToWithAttributes($method, $uri, TableNode $table)
    {
        foreach( $table->getTable() as $row ){
            $this->addJsonApiAttribute($row[0], $row[1]);
        }

        $this->iSendARequestWithBearerTo($method, $uri);
    }

    /**
     * @Then /^The response code should be "([^"]*)"$/
     *
     * @param $responseCode
     * @throws \Exception
     */
    public function theResponseCodeShouldBe($responseCode)
    {
        PHPUnit_Framework_Assert::assertEquals($responseCode, $this->response->getStatusCode());
    }

    /**
     * @Then /^The response content type should be "([^"]*)"$/
     *
     * @param $contentType
     * @throws \Exception
     */
    public function theResponseContentTypeShouldBe($contentType)
    {
        PHPUnit_Framework_Assert::assertEquals($contentType, $this->response->headers->get('content-type'));
    }

    /**
     * @Then /^The json response should have a "([^"]*)" key$/
     */
    public function theJsonResponseShouldHaveAKey($key)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        $array = get_object_vars($this->json_response);

        foreach( $keys as $index => $key ){
            PHPUnit_Framework_Assert::assertArrayHasKey($key, $array);
            $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
        }
    }

    /**
     * @Then /^The json response should have a key "([^"]*)" with value "([^"]*)"$/
     */
    public function theJsonResponseShouldHaveAKeyWithValue($key, $value)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        $array = get_object_vars($this->json_response);

        foreach( $keys as $index => $key ){
            PHPUnit_Framework_Assert::assertArrayHasKey($key, $array);
            $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
            if ( $index == count($keys) - 1 ){
                PHPUnit_Framework_Assert::assertEquals($value, $array);
            }
        }
    }

    /**
     * @Then /^The json response should have a key "([^"]*)" with placeholder value "([^"]*)"$/
     */
    public function theJsonResponseShouldHaveAKeyWithPlaceholderValue($key, $placeholder)
    {
        $value = $this->getPlaceholder($placeholder);

        $this->theJsonResponseShouldHaveAKeyWithValue($key, $value);
    }


    /**
     * @Then /^The response should be valid json api$/
     */
    public function theResponseShouldBeValidJsonApi()
    {
        $valid = false;

        //A document MUST contain at least one of the following top-level members:
        if ( isset( $this->json_response->data ) ) $valid = true;
        if ( isset( $this->json_response->errors ) ) $valid = true;
        if ( isset( $this->json_response->meta ) ) $valid = true;

        PHPUnit_Framework_Assert::assertTrue($valid, "The response does not contain one of the following fields: data, errors, meta");

        $this->theResponseContentTypeShouldBe('application/vnd.api+json');
    }

    /**
     * @When /^I send a "([^"]*)" request to "([^"]*)" with following form fields$/
     */
    public function iSendARequestToWithFollowingFormFields($method, $uri, TableNode $table)
    {
        foreach( $table->getTable() as $row ){
            $this->addFormField($row[0], $row[1]);
        }

        $this->sendRequest($method, $uri);
    }

    /**
     * Prints last response body.
     *
     * @Then print response
     */
    public function printResponse()
    {
        $request = $this->client->getRequest();
        $response = $this->response;

        echo sprintf(
            "%s %s => %d:\n%s",
            $request->getMethod(),
            $request->getRequestUri(),
            $response->getStatusCode(),
            (string) $response->getContent()
        );
    }

    /**
     * @Then print json response
     */
    public function printJsonResponse()
    {
        $request = $this->client->getRequest();
        $response = $this->response;

        echo sprintf(
            "%s %s => %d:\n%s",
            $request->getMethod(),
            $request->getRequestUri(),
            $response->getStatusCode(),
            (string) print_r(json_decode($response->getContent()), true)
        );
    }

}
