<?php
namespace Bagaar\BehatMinkApi;

use Behat\Behat\Tester\Exception\PendingException;
use Behat\Behat\Context\Context;
use Behat\Behat\Context\SnippetAcceptingContext;
use Behat\Gherkin\Node\PyStringNode;
use Behat\Gherkin\Node\TableNode;
use Behat\MinkExtension\Context\MinkContext;
use PHPUnit_Framework_Assert;


/**
 * Defines application features from the specific context.
 */
class ApiBaseFeatureContext extends MinkContext implements Context, SnippetAcceptingContext
{
    use \Laracasts\Behat\Context\Migrator;
    use \Laracasts\Behat\Context\DatabaseTransactions;

    /**
     * @var \Symfony\Component\HttpKernel\Client
     */
    protected $client;

    /**
     * @var Symfony\Component\HttpFoundation\Response
     */
    protected $response;

    protected $placeholders = [
        'LAST_ID'
    ];
    protected $dynamicPlaceholders = [];

    protected $oauth_access_token;
    protected $add_header_bearer = false;

    protected $json_response;

    protected $form_fields = [];
    protected $json_api_attributes = [];
    protected $api_attributes = [];

    protected $last_id;

    private function validateBody($body)
    {
        foreach( $this->placeholders as $ph ){
            $body = str_replace($ph, $this->getPlaceholder($ph), $body);
        }

        return $body;
    }


    private function validateUri($uri){

        foreach( $this->placeholders as $ph ){
            $uri = str_replace($ph, $this->getPlaceholder($ph), $uri);
        }

        if ( strpos($uri, 'http') === 0 ){
            return $uri;
        }

        if ( $uri[0] != '/' ) $uri = '/' . $uri;
        $uri = env('APP_URL') . $uri;

        return $uri;
    }

    private function parseJsonResponse()
    {
        $this->json_response = json_decode($this->response->getContent());
    }

    private function parseKey($key)
    {
        return explode('.', $key);
    }

    private function addFormField($key, $value)
    {
        $this->form_fields[$key] = $value;
    }

    private function addJsonApiAttribute($key, $value)
    {
        $this->json_api_attributes[$key] = $value;
    }

    private function addApiAttribute($key, $value)
    {
        $this->api_attributes[$key] = $value;
    }

    /**
     * @return \Symfony\Component\HttpKernel\Client
     */
    protected function getClient()
    {
        $this->client = $this->getSession()->getDriver()->getClient();
        return $this->client;
    }

    /**
     * @param $method
     * @param $uri
     * @param array $parameters
     * @param array $files
     * @param array $server
     * @param null $content
     * @param bool|true $changeHistory
     */
    protected function sendRequest($method, $uri, $parameters = [], $files = [], $server = [], $content = null, $changeHistory = true)
    {
        $this->getClient();
        $this->client->followRedirects(false);

        if ( ! isset( $server['HTTP_Accept'] ) ) $server['HTTP_Accept'] = 'application/json';

        $uri = $this->validateUri($uri);
        $content = $this->validateBody($content);

        if ( $this->add_header_bearer && $this->oauth_access_token ){
            $server['HTTP_Authorization'] = 'Bearer ' . $this->oauth_access_token;
        }

        if ( ! $content && count( $this->form_fields ) ){
            $content = http_build_query( $this->form_fields, null, '&' );
            $parameters = $this->form_fields;

            $this->form_fields = [];

            if ( ! isset($server['CONTENT_TYPE']) ) $server['CONTENT_TYPE'] = 'application/x-www-form-urlencoded';
        }

        if ( ! $content && count( $this->json_api_attributes ) ){
            $json = [
                'data' => [
                    'attributes' => $this->json_api_attributes
                ]
            ];
            $content = json_encode($json);

            $this->json_api_attributes = [];

            if ( ! isset($server['CONTENT_TYPE']) ) $server['CONTENT_TYPE'] = 'application/vnd.api+json';
        }

        if ( ! $content && count( $this->api_attributes ) ){

            $content = json_encode($this->api_attributes);
            $this->api_attributes = [];

            if ( ! isset($server['CONTENT_TYPE']) ) $server['CONTENT_TYPE'] = 'application/json';
        }

        $this->client->request($method, $uri, $parameters, $files, $server, $content, $changeHistory);

        $this->response = $this->client->getResponse();
    }


    /**
     * @param $username
     * @param $password
     */
    protected function oauthAuthentication($username, $password, $clientId = null, $clientSecret = null)
    {
        $request = [
            'username' => $username,
            'password' => $password,
            'grant_type' => 'password'
        ];

        if ($clientId && $clientSecret) {
            $request['client_id'] = $clientId;
            $request['client_secret'] = $clientSecret;
        }

        $this->sendRequest('POST', 'oauth/token', [], [], ['CONTENT_TYPE' => 'application/json'], json_encode($request));

        $this->theResponseCodeShouldBe(200);
        $this->theJsonResponseShouldHaveAKey('access_token');
        $this->theJsonResponseShouldHaveAKey('token_type');
        $this->theJsonResponseShouldHaveAKey('expires_in');
        $this->theJsonResponseShouldHaveAKey('refresh_token');

        $this->parseJsonResponse();

        $this->oauth_access_token = $this->json_response->access_token;
    }

    protected function getPlaceholder( $placeholder )
    {
        if ( isset($this->dynamicPlaceholders[$placeholder]) ) return $this->dynamicPlaceholders[$placeholder];

        switch( $placeholder ){
            case 'LAST_ID':
                return $this->last_id;
                break;
        }
    }


    /**
     * @Then /^I authenticate with username "([^"]*)" and password "([^"]*)" and client_id "([^"]*)" and client_secret "([^"]*)"$/
     * @param $username
     * @param $password
     * @param $clientId
     * @param $clientSecret
     */
    public function iAuthenticateWithUsernameAndPasswordAndClientIdAndClientSecret($username, $password, $clientId, $clientSecret)
    {
        foreach( $this->placeholders as $ph ){
            $clientId = str_replace($ph, $this->getPlaceholder($ph), $clientId);
            $clientSecret = str_replace($ph, $this->getPlaceholder($ph), $clientSecret);
        }

        $this->oauthAuthentication($username, $password, $clientId, $clientSecret);
    }

    /**
     * @Then /^I authenticate with username "([^"]*)" and password "([^"]*)"$/
     * @param $username
     * @param $password
     */
    public function iAuthenticateWithUsernameAndPassword($username, $password)
    {
        $this->oauthAuthentication($username, $password);
    }


    /**
     * @When /^I send a "([^"]*)" request to "([^"]*)"$/
     *
     * @param $method
     * @param $uri
     */
    public function iSendARequestTo($method, $uri)
    {
        $this->sendRequest($method, $uri);
    }

    /**
     * @When /^I send a "([^"]*)" request with bearer to "([^"]*)"$/
     */
    public function iSendARequestWithBearerTo($method, $uri)
    {
        $this->add_header_bearer = true;
        $this->sendRequest($method, $uri);
        $this->add_header_bearer = false;
    }

    /**
     * @When /^I send a "([^"]*)" request with bearer to "([^"]*)" with attributes$/
     */
    public function iSendARequestWithBearerToWithAttributes($method, $uri, TableNode $table)
    {
        foreach( $table->getTable() as $row ){
            $this->addApiAttribute($row[0], $row[1]);
        }

        $this->iSendARequestWithBearerTo($method, $uri);
    }

    /**
     * @When /^I send a "([^"]*)" json api request with bearer to "([^"]*)"$/
     */
    public function iSendAJsonApiRequestWithBearerTo($method, $uri)
    {
        $this->iSendARequestWithBearerTo($method, $uri);

        $this->parseJsonResponse();

        $this->theResponseShouldBeValidJsonApi();
    }

    /**
     * @When /^I send a "([^"]*)" json api request to "([^"]*)" with attributes$/
     */
    public function iSendAJsonApiRequestToWithAttributes($method, $uri, TableNode $table)
    {
        foreach( $table->getTable() as $row ){
            $this->addJsonApiAttribute($row[0], $row[1]);
        }

        $this->iSendARequestWithBearerTo($method, $uri);
    }


    /**
     * @When /^I send a "([^"]*)" json api request with bearer to "([^"]*)" with attributes$/
     */
    public function iSendAJsonApiRequestWithBearerToWithAttributes($method, $uri, TableNode $table)
    {
        foreach( $table->getTable() as $row ){
            $this->addJsonApiAttribute($row[0], $row[1]);
        }

        $this->iSendARequestWithBearerTo($method, $uri);
    }

    /**
     * @Then /^The response code should be "([^"]*)"$/
     *
     * @param $responseCode
     * @throws \Exception
     */
    public function theResponseCodeShouldBe($responseCode)
    {
        PHPUnit_Framework_Assert::assertEquals($responseCode, $this->response->getStatusCode());
    }


    /**
     * @Then /^The response header should contain "([^"]*)" with value "([^"]*)"$/
     */
    public function theResponseHeaderShouldContainWithValue($key, $value)
    {
        foreach( $this->placeholders as $ph ){
            $value = str_replace($ph, $this->getPlaceholder($ph), $value);
        }

        PHPUnit_Framework_Assert::assertEquals($value, $this->response->headers->get($key));
    }


    /**
     * @Then /^The response content type should be "([^"]*)"$/
     *
     * @param $contentType
     * @throws \Exception
     */
    public function theResponseContentTypeShouldBe($contentType)
    {
        PHPUnit_Framework_Assert::assertEquals($contentType, $this->response->headers->get('content-type'));
    }

    /**
     * @Then /^The json response should have a "([^"]*)" key$/
     */
    public function theJsonResponseShouldHaveAKey($key)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        if ( is_array($this->json_response) ){
            $array = $this->json_response;
        } else {
            $array = get_object_vars($this->json_response);
        }

        foreach( $keys as $index => $key ){
            PHPUnit_Framework_Assert::assertArrayHasKey($key, $array);
            $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
        }
    }


    /**
     * @Then /^The json response should not have a "([^"]*)" key$/
     */
    public function theJsonResponseShouldNotHaveAKey($key)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        if ( is_array($this->json_response) ){
            $array = $this->json_response;
        } else {
            $array = get_object_vars($this->json_response);
        }

        $found = false;

        foreach( $keys as $index => $key ){
            $found = array_key_exists($key, $array);
            if ( isset( $array[$key] ) ){
                $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
            }
        }

        PHPUnit_Framework_Assert::assertFalse($found);
    }



    /**
     * @Then /^The json response should have a key "([^"]*)" with value "([^"]*)"$/
     */
    public function theJsonResponseShouldHaveAKeyWithValue($key, $value)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        if ( is_array($this->json_response) ){
            $array = $this->json_response;
        } else {
            $array = get_object_vars($this->json_response);
        }

        foreach( $keys as $index => $key ){
            PHPUnit_Framework_Assert::assertArrayHasKey($key, $array);
            $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
            if ( $index == count($keys) - 1 ){
                PHPUnit_Framework_Assert::assertEquals($value, $array);
            }
        }
    }


    /**
     * @Then /^The json response should have a key "([^"]*)" with value less than "([^"]*)"$/
     */
    public function theJsonResponseShouldHaveAKeyWithValueLessThan($key, $value)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        if ( is_array($this->json_response) ){
            $array = $this->json_response;
        } else {
            $array = get_object_vars($this->json_response);
        }

        foreach( $keys as $index => $key ){
            PHPUnit_Framework_Assert::assertArrayHasKey($key, $array);
            $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
            if ( $index == count($keys) - 1 ){
                PHPUnit_Framework_Assert::assertLessThan($value, $array);
            }
        }
    }


    /**
     * @Then /^The json response should have a key "([^"]*)" with placeholder value "([^"]*)"$/
     */
    public function theJsonResponseShouldHaveAKeyWithPlaceholderValue($key, $placeholder)
    {
        $value = $this->getPlaceholder($placeholder);

        $this->theJsonResponseShouldHaveAKeyWithValue($key, $value);
    }


    /**
     * @Then /^The json response should be an array with length "([^"]*)"$/
     */
    public function theJsonResponseShouldBeAnArrayWithLength($length)
    {
        $this->parseJsonResponse();

        PHPUnit_Framework_Assert::assertInternalType('array', $this->json_response);

        PHPUnit_Framework_Assert::assertEquals((int)$length, count($this->json_response));
    }


    /**
     * @Then /^The json response key "([^"]*)" should be an array with length "([^"]*)"$/
     */
    public function theJsonResponseKeyShouldBeAnArrayWithLength($key, $length)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        if ( is_array($this->json_response) ){
            $array = $this->json_response;
        } else {
            $array = get_object_vars($this->json_response);
        }

        foreach( $keys as $index => $key ){
            PHPUnit_Framework_Assert::assertArrayHasKey($key, $array);
            $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
            if ( $index == count($keys) - 1 ){
                //PHPUnit_Framework_Assert::assertEquals($value, $array);
                PHPUnit_Framework_Assert::assertInternalType('array', $array);
                PHPUnit_Framework_Assert::assertEquals((int)$length, count($array));
            }
        }

    }

    /**
     * @Then /^The json response key "([^"]*)" should be an array with length less than "([^"]*)"$/
     */
    public function theJsonResponseKeyShouldBeAnArrayWithLengthLessThan($key, $length)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        if ( is_array($this->json_response) ){
            $array = $this->json_response;
        } else {
            $array = get_object_vars($this->json_response);
        }

        foreach( $keys as $index => $key ){
            PHPUnit_Framework_Assert::assertArrayHasKey($key, $array);
            $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
            if ( $index == count($keys) - 1 ){
                //PHPUnit_Framework_Assert::assertEquals($value, $array);
                PHPUnit_Framework_Assert::assertInternalType('array', $array);
                PHPUnit_Framework_Assert::assertLessThan((int)$length, count($array));
            }
        }

    }


    /**
     * @Then /^The json response should be an array with length less than "([^"]*)"$/
     */
    public function theJsonResponseShouldBeAnArrayWithLengthLessThan($length)
    {
        $this->parseJsonResponse();

        PHPUnit_Framework_Assert::assertInternalType('array', $this->json_response);

        PHPUnit_Framework_Assert::assertLessThan((int)$length, count($this->json_response));
    }


    /**
     * @Then /^Remember the json response key "([^"]*)" as placeholder "([^"]*)"$/
     */
    public function rememberTheJsonResponseKeyAsPlaceholder($key, $placeholder)
    {
        $this->parseJsonResponse();

        $keys = $this->parseKey($key);

        if ( is_array($this->json_response) ){
            $array = $this->json_response;
        } else {
            $array = get_object_vars($this->json_response);
        }

        foreach( $keys as $index => $key ){
            PHPUnit_Framework_Assert::assertArrayHasKey($key, $array);
            $array = is_object($array[$key]) ? get_object_vars($array[$key]) : $array[$key];
            if ( $index == count($keys) - 1 ){
                $this->dynamicPlaceholders[$placeholder] = $array;
                $this->placeholders[] = $placeholder;

                echo $placeholder . ' => ' . $array;
            }
        }
    }


    /**
     * @Then /^The response should be valid json api$/
     */
    public function theResponseShouldBeValidJsonApi()
    {
        $valid = false;

        //A document MUST contain at least one of the following top-level members:
        if ( isset( $this->json_response->data ) ) $valid = true;
        if ( isset( $this->json_response->errors ) ) $valid = true;
        if ( isset( $this->json_response->meta ) ) $valid = true;

        PHPUnit_Framework_Assert::assertTrue($valid, "The response does not contain one of the following fields: data, errors, meta");

        $this->theResponseContentTypeShouldBe('application/vnd.api+json');
    }

    /**
     * @When /^I send a "([^"]*)" request to "([^"]*)" with following form fields$/
     */
    public function iSendARequestToWithFollowingFormFields($method, $uri, TableNode $table)
    {
        foreach( $table->getTable() as $row ){
            $this->addFormField($row[0], $row[1]);
        }

        $this->sendRequest($method, $uri);
    }


    /**
     * @When /^I send a "([^"]*)" json request to "([^"]*)" with following body$/
     */
    public function iSendARequestToWithFollowingBody($method, $uri, PyStringNode $string)
    {
        $this->sendRequest($method, $uri, [], [], ['CONTENT_TYPE' => 'application/json'], (string)$string);
    }


    /**
     * Prints last response body.
     *
     * @Then print response
     */
    public function printResponse()
    {
        $request = $this->client->getRequest();
        $response = $this->response;

        echo sprintf(
            "%s %s => %d:\n%s",
            $request->getMethod(),
            $request->getRequestUri(),
            $response->getStatusCode(),
            (string) $response->getContent()
        );
    }

    /**
     * @Then print json response
     */
    public function printJsonResponse()
    {
        $request = $this->client->getRequest();
        $response = $this->response;

        echo sprintf(
            "%s %s => %d:\n%s",
            $request->getMethod(),
            $request->getRequestUri(),
            $response->getStatusCode(),
            (string) print_r(json_decode($response->getContent()), true)
        );
    }


    /**
     * @Then /^The model "([^"]*)" with id "([^"]*)" should have a key "([^"]*)" which is not empty$/
     */
    public function theModelWithIdShouldHaveAKeyWhichIsNotEmpty($model, $id, $attr)
    {
        foreach( $this->placeholders as $ph ){
            $id = str_replace($ph, $this->getPlaceholder($ph), $id);
        }

        $model_instance = app('\\App\\Models\\' . $model);
        $result = $model_instance->find($id);

        PHPUnit_Framework_Assert::assertNotEmpty($result->{$attr});
    }

    /**
     * @Given /^The model "([^"]*)" has "([^"]*)" entries$/
     */
    public function theModelHasEntries($model, $count)
    {
        $models = factory('App\\Models\\' . $model, (int)$count)->create();

        $this->last_id = $models[ count($models) - 1 ]->id;
    }

}
