<?php

declare(strict_types=1);

namespace Bagaar\LaravelAzureBlobStorage;

use Illuminate\Contracts\Filesystem\Filesystem;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use InvalidArgumentException;

class AzureBlobStorageManager
{
    private array $connections;

    public function __construct(array $connections = [])
    {
        $this->connections = $connections;
    }

    public function connect(string|array|null $connection = null): ContainerFilesystemFactory
    {
        if (is_array($connection)) {
            return ContainerFilesystemFactory::connect($this->createConnection($connection));
        }

        if (!is_array(Arr::get($this->connections, $connection ??= 'default'))) {
            throw new InvalidArgumentException("Azure Blob Storage connection [{$connection}] is not configured.");
        }

        return ContainerFilesystemFactory::connect($this->createConnection((array) $this->connections[$connection]));
    }

    public function container(string $path, string $pathPrefix = ''): Filesystem
    {
        return $this->connect()->container($path, $pathPrefix);
    }

    private function createConnection(array $config): Connection
    {
        return Connection::make($this->convertSnakeCaseKeysToPascalCase($config));
    }

    private function convertSnakeCaseKeysToPascalCase(array $config): array
    {
        return collect($config)->mapWithKeys(function ($value, $key) {
            return [(string) Str::of($key)->camel()->ucfirst() => $value];
        })->toArray();
    }
}
