<?php

declare(strict_types=1);

namespace Bagaar\LaravelAzureBlobStorage;

use Illuminate\Contracts\Filesystem\Filesystem as FilesystemContract;
use Illuminate\Filesystem\FilesystemAdapter;
use League\Flysystem\Filesystem;
use MicrosoftAzure\Storage\Blob\BlobRestProxy;
use MicrosoftAzure\Storage\Common\Internal\StorageServiceSettings;

final class ContainerFilesystemFactory
{
    private Connection $connection;

    private function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    public static function connect(Connection $connection): self
    {
        return new self($connection);
    }

    /**
     * Create a blob storage filesystem for a given container and optional path prefix.
     */
    public function container(string $container, string $pathPrefix = ''): FilesystemContract
    {
        $adapter = new AzureBlobStorageAdapter(
            client: BlobRestProxy::createBlobService($this->connection->toString()),
            container: $container,
            prefix: $pathPrefix,
            mimeTypeDetector: null,
            maxResultsForContentsListing: 5000,
            visibilityHandling: AzureBlobStorageAdapter::ON_VISIBILITY_THROW_ERROR,
            serviceSettings: StorageServiceSettings::createFromConnectionString($this->connection->toString()),
        );

        $filesystem = new Filesystem($adapter);

        return new FilesystemAdapter($filesystem, $adapter);
    }
}
