<?php

namespace Bagaar\LaravelAzureStorage\Providers;

use DateTimeInterface;
use League\Flysystem\AzureBlobStorage\AzureBlobStorageAdapter as BaseAzureBlobStorageAdapter;
use League\MimeTypeDetection\MimeTypeDetector;
use MicrosoftAzure\Storage\Blob\BlobRestProxy;
use MicrosoftAzure\Storage\Blob\BlobSharedAccessSignatureHelper;
use MicrosoftAzure\Storage\Common\Internal\Resources;

class AzureBlobStorageAdapter extends BaseAzureBlobStorageAdapter
{
    /** Azure Blob Client
     * @var BlobRestProxy */
    private BlobRestProxy $client;

    /** Container name - public or private
     * @var string */
    private string $container;

    /** Container key
     * @var string */
    private string $key;

    /** TTL for SAS URLs
     * @var int */
    private int $sasExpiryInMinutes;

    /**
     * Constructor.
     *
     * @param  BlobRestProxy  $client
     * @param  string  $container - public or private
     * @param  int  $sasExpiryInMinutes
     * @param  string  $prefix
     * @param  MimeTypeDetector|null  $mimeTypeDetector
     * @param  int  $maxResultsForContentsListing
     * @param  string  $visibilityHandling
     */
    public function __construct(
        BlobRestProxy $client,
        string $container,
        string $key,
        int $sasExpiryInMinutes = 60,
        string $prefix = '',
        MimeTypeDetector $mimeTypeDetector = null,
        int $maxResultsForContentsListing = 5000,
        string $visibilityHandling = self::ON_VISIBILITY_THROW_ERROR,
    ) {
        parent::__construct($client, $container, $prefix, $mimeTypeDetector, $maxResultsForContentsListing, $visibilityHandling);

        $this->sasExpiryInMinutes = $sasExpiryInMinutes;
        $this->client = $client;
        $this->container = $container;
        $this->key = $key;
    }

    public function getUrl($path)
    {
        return $this->url($path);
    }

    public function getTemporaryUrl(string $path, DateTimeInterface $expiration, array $options = []): string
    {
        return $this->temporaryUrl($path, $expiration, $options);
    }

    /**
     * Returns a url to the file at the given path.
     *
     * @param  string  $path
     * @return string
     */
    public function url(string $path): string
    {
        return $this->client->getBlobUrl($this->container, $path);
    }

    /**
     * Returns a url with SAS/expiration date to the file at the given path.
     *
     * @param  string  $path
     * @param  DateTimeInterface  $expiration
     * @param  array  $options
     * @return string
     */
    public function temporaryUrl(string $path, DateTimeInterface $expiration, array $options = []): string
    {
        $helper = new BlobSharedAccessSignatureHelper(
            $this->client->getAccountName(),
            $this->key
        );

        $blobPath = (! empty($this->container)) ? $this->container.'/'.$path : $path;

        $formattedExpiryTime = $expiration->format('Y-m-d\TH:i:s\Z');
        $sasToken = $helper->generateBlobServiceSharedAccessSignatureToken(
            Resources::RESOURCE_TYPE_BLOB,
            $blobPath,
            'r', // Read only
            $formattedExpiryTime // A valid ISO 8601 format expiry time (ex: 2020-07-08T09:59:31Z)
        );

        $connectionString =
            Resources::BLOB_ENDPOINT_NAME.'='.'https://'.$this->client->getAccountName().'.'.Resources::BLOB_BASE_DNS_NAME.';'.
            Resources::SAS_TOKEN_NAME.'='.$sasToken;
        $blobClientWithSAS = BlobRestProxy::createBlobService($connectionString);

        $blobUrlWithSAS = sprintf(
            '%s%s?%s',
            (string) $blobClientWithSAS->getPsrPrimaryUri(),
            $blobPath,
            $sasToken
        );

        return $blobUrlWithSAS;
    }
}
