<?php

namespace Tests\Unit;

use Bagaar\LaravelAzureStorage\Helpers\PrivateStorageHelper;
use Carbon\Carbon;
use Carbon\CarbonImmutable;
use DateInterval;
use Illuminate\Foundation\Bootstrap\LoadEnvironmentVariables;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Tests\TestCase;

class StorageTest extends TestCase
{
    /**
     * Define environment setup.
     *
     * @param  \Illuminate\Foundation\Application  $app
     * @return void
     */
    protected function defineEnvironment($app)
    {
        // load our .env.testing
        $app->useEnvironmentPath(__DIR__.'/../..');
        $app->bootstrapWith([LoadEnvironmentVariables::class]);

        // create config
        $app['config']->set('filesystems.default', 'azure-public');
        $app['config']->set('filesystems.disks.azure-public', [
            'driver' => 'azure',
            'name' => env('AZURE_PUBLIC_STORAGE_NAME', 'public'),
            'key' => env('AZURE_PUBLIC_STORAGE_KEY'),
            'endpointSuffix' => env('AZURE_ENDPOINT_SUFFIX', 'core.windows.net'),
            'container' => env('AZURE_PUBLIC_STORAGE_CONTAINER'),
            'url' => ('https://'
                .env('AZURE_PUBLIC_STORAGE_NAME')
                .'.blob.'.env('AZURE_ENDPOINT_SUFFIX').'/'
                .env('AZURE_PUBLIC_STORAGE_CONTAINER').'/'
            ),
            'sasExpiryMinutes' => 0,
            'throw' => true,
        ]);
        $app['config']->set('filesystems.disks.azure-private', [
            'driver' => 'azure',
            'name' => env('AZURE_PRIVATE_STORAGE_NAME'),
            'key' => env('AZURE_PRIVATE_STORAGE_KEY'),
            'endpointSuffix' => env('AZURE_ENDPOINT_SUFFIX', 'core.windows.net'),
            'container' => env('AZURE_PRIVATE_STORAGE_CONTAINER', null),
            'url' => ('https://'
                .env('AZURE_PRIVATE_STORAGE_NAME')
                .'.blob.'.env('AZURE_ENDPOINT_SUFFIX').'/'
                .env('AZURE_PRIVATE_STORAGE_CONTAINER').'/'
            ),
            'sasExpiryMinutes' => env('AZURE_PRIVATE_STORAGE_EXPIRY_MINUTES', 60),
            'throw' => true,
        ]);
    }

    /**
     * @param  Application  $app
     * @return array
     */
    protected function getPackageProviders($app)
    {
        return ['Bagaar\LaravelAzureStorage\Providers\AzureBlobStorageServiceProvider'];
    }

    protected function setUp(): void
    {
        parent::setUp();

        // fix timestamps for testing
        $this->now = CarbonImmutable::now()->micro(0);
        Carbon::setTestNow($this->now);
    }

    public function testStorageUrlGeneration()
    {
        // test public storage url generation
        $publicAzureStorage = Storage::disk('azure-public');
        $publicUrl = $publicAzureStorage->url('uploads/uuid.jpg');
        $this->assertEquals(config('filesystems.disks.azure-public.url').'uploads/uuid.jpg', $publicUrl);

        // test private storage temporary url with SAS generation
        $privateAzureStorage = Storage::disk('azure-private');
        $expirationDate = ($this->now)->add(new DateInterval('PT'.'60'.'M'));
        $privateUrl = $privateAzureStorage->temporaryUrl('uploads/uuid.jpg', $expirationDate);

        $privateUrlParts = explode('?', $privateUrl);
        $actualBaseUrl = $privateUrlParts[0];
        $privateUrlParamParts = explode('&', $privateUrlParts[1]);
        foreach ($privateUrlParamParts as $part) {
            if (Str::startsWith($part, 'se=')) {
                $actualExpiryDateSAS = explode('=', $part)[1];
            }
        }

        $expectedBaseUrl = config('filesystems.disks.azure-private.url').'uploads/uuid.jpg';
        $expectedExpiryDateSAS = $expirationDate->format('Y-m-d\TH:i:s\Z');

        $this->assertEquals($expectedBaseUrl, $actualBaseUrl);
        $this->assertEquals($expectedExpiryDateSAS, $actualExpiryDateSAS);
    }

    public function testStorageUrlHelper()
    {
        $privateUrl = PrivateStorageHelper::getTemporaryUrl('azure-private', 'uploads/uuid.jpg');

        $privateUrlParts = explode('?', $privateUrl);
        $actualBaseUrl = $privateUrlParts[0];
        $privateUrlParamParts = explode('&', $privateUrlParts[1]);
        foreach ($privateUrlParamParts as $part) {
            if (Str::startsWith($part, 'se=')) {
                $actualExpiryDateSAS = explode('=', $part)[1];
            }
        }

        $expectedBaseUrl = config('filesystems.disks.azure-private.url').'uploads/uuid.jpg';
        $expectedExpiryDateSAS = ($this->now)->add(
            new DateInterval('PT'.config('filesystems.disks.azure-private.sasExpiryMinutes').'M')
        )->format('Y-m-d\TH:i:s\Z');

        $this->assertEquals($expectedBaseUrl, $actualBaseUrl);
        $this->assertEquals($expectedExpiryDateSAS, $actualExpiryDateSAS);
    }

    public function testStorageUrlHelperReturnsNullIfPathNull()
    {
        $value = PrivateStorageHelper::getTemporaryUrl('azure-private', null);
        $this->assertNull($value);
    }

    protected function tearDown(): void
    {
        Carbon::setTestNow(null);

        parent::tearDown();
    }
}
