<?php

namespace Bagaar\LaravelAzureStorage\Helpers;

use DateInterval;
use DateTimeInterface;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;

class PrivateStorageHelper
{
    /**
     * Returns a url with optional SAS/expiration date to the file at the given path.
     * If no expirationDate is given, create it from now() with the default env settings.
     * If the Storage adapter does not support temporary urls (like local in testing envs),
     * return a regular URL.
     *
     * @param  string|null  $path
     * @param  DateTimeInterface|null  $expirationDate
     * @return string|null
     */
    public static function getTemporaryUrl(string $storageDiskName, ?string $path, DateTimeInterface $expirationDate = null): ?string
    {
        if(!$path) {
            return null;
        }
        $storage = Storage::disk($storageDiskName);
        $adapterSupportsTempUrls = method_exists($storage->getAdapter(), 'temporaryUrl');

        if ($adapterSupportsTempUrls) {
            /** @var \App\Providers\AzureBlobStorageAdapter $storage */
            if (! $expirationDate) {
                $minutes = config('filesystems.disks.'.$storageDiskName.'.sasExpiryMinutes') ?? 60;
                $expirationDate = (Carbon::now())->add(new DateInterval('PT'.$minutes.'M'));
            }

            return $storage->temporaryUrl($path, $expirationDate);
        }

        return $storage->url($path);
    }
}
