# Laravel File Uploads

## Usage

### Upload a file

#### Endpoint

The endpoint for uploading a file is configured in the `laravel-file-uploads.php` config file.

It defaults to `/uploads`.

The endpoint expects a multipart form data request with a file field named `file`.

```javascript
async function sendData(url, data) {
  const formData  = new FormData();
  formData.append('file', data);

  const response = await fetch(url, {
    method: 'POST',
    body: formData
  });
}
```

The response is a JSON object with the following structure:

```json
{
    "id": "9820c4bb-3693-40cb-b1d9-1c634aeaa47c",
    "updated_at": "2023-01-02T13:11:09.000000Z",
    "created_at": "2023-01-02T13:11:09.000000Z",
    "path": "uploads/9820c4bb-3693-40cb-b1d9-1c634aeaa47c",
    "original_name": "lemon.png",
    "mime_type": "image/png",
    "size": 29598
}
```

The `id` is the UUID of the uploaded file. It can be used to reference the file in other parts of the application.

#### Middleware

Optionally, you can add middlewares to the endpoint.

This is configured in the config file through the `middlewares` key.

#### Validation

By default, there is file_size validation and mime_type validation.

- The `size` validation is configured in the config file through the `max_filesize` key and defaults to `upload_max_filesize` from php.ini.
- The `mimes` validation is configured in the config file through the `allowed_mimetypes` key and defaults to `*`.
- More validation rules can be added with the `extra_validation_rules` key in the config file.

#### Storage

The storage disk is configured in the config file through the `disk` key.
Optionally, you can specify a path prefix for the uploaded files through the `path_prefix` key.

### Link an uploaded file to a model

You can use the `@linkUpload` directive in your graphql schema to indicate that a field is a reference to an uploaded file.
With the attribute you can specify allowed mime types, max filesize and if the filesize should be saved on the model as well.

```graphql
"""
Link uploaded files to a model using a UUID.
arguments:
allowedMimeTypes: list of allowed mime types (["image/jpeg", "image/png", ...]) (optional) (default: all mime types allowed)
maxFileSize: maximum file size in bytes (optional) (default: 64MB)
storeFileSize: store the file size in the database (optional) (default: false)
fileSizeColumn: name of the column in the database that stores the file size (optional) (defaults a column with the same name and the suffix "_filesize")
"""            
directive @linkUpload(
allowedMimeTypes: [String!]
maxFileSize: Int
storeFileSize: Boolean
fileSizeColumn: String
) on ARGUMENT_DEFINITION | INPUT_FIELD_DEFINITION | FIELD_DEFINITION
```

for example: 

```graphql
input UserInput {
    name: String!
    avatar: String @linkUpload(storeFileSize: true)
}

type Mutation {
    createUser(input: UserInput! @spread): User @create
}
```
