<?php

namespace Bagaar\LaravelFileUploads\GraphQL\Directives;

use Bagaar\LaravelFileUploads\Models\Upload;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Nuwave\Lighthouse\Exceptions\ValidationException;
use Nuwave\Lighthouse\Execution\Arguments\Argument;
use Nuwave\Lighthouse\Execution\Arguments\ArgumentSet;
use Nuwave\Lighthouse\Execution\ResolveInfo;
use Nuwave\Lighthouse\Schema\Directives\BaseDirective;
use Nuwave\Lighthouse\Schema\Values\FieldValue;
use Nuwave\Lighthouse\Support\Contracts\FieldMiddleware;
use Nuwave\Lighthouse\Support\Contracts\GraphQLContext;

class LinkUploadDirective extends BaseDirective implements FieldMiddleware
{
    private bool $withFilesize = false;

    public static function definition(): string
    {
        return /** @lang GraphQL */ <<<GRAPHQL
"""
Link uploaded files to a model using a UUID.
arguments:
allowedMimeTypes: list of allowed mime types ("image/jpeg", "image/png", etc.) (optional)
maxFileSize: maximum file size in bytes (optional)
storeFileSize: store the file size in the database (default: false)
fileSizeColumn: name of the column in the database that stores the file size (defaults a column with the same name and the suffix "_filesize")
"""            
directive @linkUpload(
allowedMimeTypes: [String!]
maxFileSize: Int
storeFileSize: Boolean
fileSizeColumn: String
) on ARGUMENT_DEFINITION | INPUT_FIELD_DEFINITION | FIELD_DEFINITION
GRAPHQL;
    }

    public function handleField(FieldValue $fieldValue, \Closure $next)
    {
        $resolver = $fieldValue->getResolver();
        $fieldValue->setResolver(
            function ($root, array $args, GraphQLContext $context, ResolveInfo $resolveInfo) use ($resolver) {
                $resolveInfo->argumentSet = $this->transformArgumentSet($resolveInfo->argumentSet);
                $arguments = $resolveInfo->argumentSet->toArray();
                return $resolver(
                    $root,
                    $arguments,
                    $context,
                    $resolveInfo
                );
            }
        );

        return $next($fieldValue);
    }

    protected function transformArgumentSet(ArgumentSet $argumentSet): ArgumentSet
    {
        $fileSizeAttributes = [];

        foreach ($argumentSet->arguments as $name => $argument) {
            $argument->directives->each(function ($directive) use (&$argument, $name, &$fileSizeAttributes) {
                if ($directive instanceof self) {
                    $allowedMimes = $directive->directiveArgValue('allowedMimeTypes', ['*']);
                    $maxFileSize = $directive->directiveArgValue('maxFileSize', 64 * 1024 * 1024);
                    $storeFilesize = $directive->directiveArgValue('storeFileSize', false);
                    $fileSizeColumn = $directive->directiveArgValue('fileSizeColumn', $name . '_filesize');
                    $upload = $this->getUpload($argument->value);
                    $this->validateUpload($upload, $allowedMimes, $maxFileSize);
                    $upload->update(['in_use' => true]);
                    $argument->value = $upload->path;
                    if ($storeFilesize) {
                        $fileSizeAttributes[$fileSizeColumn] = $upload->size;
                    }
                }
            });
        }

        foreach ($fileSizeAttributes as $name => $value) {
            $argument = new Argument();
            $argument->type = 'Int';
            $argument->value = $value;
            $argumentSet->arguments[$name] = $argument;
        }


        return $argumentSet;
    }

    private function getUpload($argumentValue): Upload
    {
        return Upload::findOrFail($argumentValue);
    }

    private function validateUpload(Upload $upload, array $allowedMimeTypes, int $maxFileSize): void
    {
        $rules = [
            'size' => 'required|integer|max:' . $maxFileSize,
        ];
        if ($allowedMimeTypes !== ['*']) {
            $rules['mime'] = 'required|string|in:' . implode(',', $allowedMimeTypes);
        }
        $validator = Validator::make(
            [
                'mime' => $upload->mime_type,
                'size' => $upload->size,
            ],
            $rules
        );

        if ($validator->fails()) {
            throw new ValidationException('Invalid upload', $validator);
        }
    }
}