<?php

namespace Tests\Feature;

use Bagaar\LaravelFileUploads\Models\Upload;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Tests\Data\Models\User;
use Tests\Data\Rules\SpecificFileNameRule;
use Tests\TestCase;

class UploadTest extends TestCase
{
    public function test_upload_file()
    {

        $response = $this->post('/uploads', [
            'file' => UploadedFile::fake()->image('avatar.jpg'),
        ]);
        $response->assertStatus(201);
        $response->assertJsonStructure([
            'id',
            'path',
            'original_name',
            'mime_type',
            'size',
            'created_at',
            'updated_at',
        ]);
    }

    public function test_upload_invalid_file()
    {
        config(['laravel-file-uploads.allowed_mimetypes' => ['image/png']]);
        $response = $this->post(
            '/uploads',
            [
                'file' => UploadedFile::fake()->image('avatar.jpg'),
            ],
            [
                'Accept' => 'application/json',
            ]
        );
        $response->assertStatus(422);
        config(['laravel-file-uploads.allowed_mimetypes' => 'image/png']);
        $response = $this->post(
            '/uploads',
            [
                'file' => UploadedFile::fake()->image('avatar.jpg'),
            ],
            [
                'Accept' => 'application/json',
            ]
        );
        $response->assertStatus(422);
    }

    public function test_upload_invalid_file_with_custom_validation_rule()
    {
        config(['laravel-file-uploads.extra_validation_rules' => [new SpecificFileNameRule('avatar.jpg')]]);
        $response = $this->post(
            '/uploads',
            [
                'file' => UploadedFile::fake()->image('foo.jpg'),
            ],
            [
                'Accept' => 'application/json',
            ]
        );
        $response->assertStatus(422);
        $response->assertJson([
            'errors' => [
                'file' => [
                    0 => 'invalid filename',
                ],
            ],
        ]);
        $response = $this->post(
            '/uploads',
            [
                'file' => UploadedFile::fake()->image('avatar.jpg'),
            ],
            [
                'Accept' => 'application/json',
            ]
        );
        $response->assertStatus(201);
    }

    public function test_link_file_to_user()
    {
        $response = $this->post(
            '/uploads',
            [
                'file' => UploadedFile::fake()->image('avatar.jpg'),
            ],
            [
                'Accept' => 'application/json',
            ]
        );
        $uploadData = $response->json();
        $response = $this->graphQL(
            'mutation ($id: String!) {
                createUser(input: {
                    name: "John Doe"
                    avatar: $id
                }) {
                    id
                    name
                    avatar
                    avatar_filesize
                }
            }',
            [
                'id' => $uploadData['id'],
            ]
        );
        $data = $response->json('data.createUser');
        $this->assertEquals('John Doe', $data['name']);
        $this->assertEquals($uploadData['path'], $data['avatar']);
        $this->assertEquals($uploadData['size'], $data['avatar_filesize']);

        $upload = Upload::firstWhere('path', $data['avatar']);
        $this->assertTrue($upload->is_used);
    }

    public function test_link_invalid_file_to_user()
    {
        $response = $this->post(
            '/uploads',
            [
                'file' => UploadedFile::fake()->image('avatar.jpg'),
            ],
            [
                'Accept' => 'application/json',
            ]
        );
        $uploadData = $response->json();
        $response = $this->graphQL(
            'mutation ($id: String!) {
                createUserRestricted(input: {
                    name: "John Doe"
                    avatar: $id
                }) {
                    id
                    name
                    avatar
                    avatar_filesize
                }
            }',
            [
                'id' => $uploadData['id'],
            ]
        );
        $data = $response->json();
        $this->assertEquals('Invalid upload', $data['errors'][0]['message']);
    }

    public function test_cleanup_abandoned_files()
    {
        $this->post(
            '/uploads',
            [
                'file' => UploadedFile::fake()->image('avatar.jpg'),
            ],
            [
                'Accept' => 'application/json',
            ]
        );
        $this->assertCount(1, Upload::all());
        DB::table('uploads')->update(['created_at' => now()->subDays(8)]);
        $this->artisan('laravel-file-uploads:cleanup');
        $this->assertCount(0, Upload::all());
    }
}
