<?php
namespace Bagaar\LaravelTemporalTables\Database\Migrations\Traits;

use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

trait TemporalMigrationTrait
{

    protected function createTemporalDateTimeFields(Blueprint $table): void
    {
        $table->dateTime('temporal_start', 6)->nullable();
        $table->dateTime('temporal_end', 6)->nullable();

        $table->index([
            'temporal_start',
            'temporal_end'
        ]);

        $table->index([
            'temporal_end'
        ]);
    }


    protected function createTemporalHistoryTable(string $originalTableName, array $triggerColumns = [])
    {
        $historyTableName = $originalTableName . '_history';

        DB::unprepared("CREATE TABLE $historyTableName LIKE $originalTableName");

        Schema::table($historyTableName, function(Blueprint $table){
            $table->dropColumn('id');
        });

        Schema::table($historyTableName, function(Blueprint $table){
            $table->id('history_id');

            $table->bigInteger('id')->unsigned()->index();
        });

        $indexes = \DB::select(\DB::raw("SHOW INDEX FROM $historyTableName"));
        foreach( $indexes as $index ){
            if ( ! $index->Non_unique && $index->Key_name != 'PRIMARY' ){
                \Schema::table($historyTableName, function (Blueprint $table) use($index){
                    $table->dropIndex($index->Key_name);
                    $table->index($index->Column_name);
                });
            }
        }

        $columns = Schema::getColumnListing($originalTableName);

        $this->createTriggerBeforeInsert($originalTableName, $historyTableName, $columns);
        $this->createTriggerAfterInsert($originalTableName, $historyTableName, $columns);

        $this->createTriggerBeforeUpdate($originalTableName, $historyTableName, $columns, $triggerColumns);

        $this->createTriggerAfterDelete($originalTableName, $historyTableName, $columns);

    }

    private function createTriggerBeforeInsert(string $originalTableName, string $historyTableName, array $columns)
    {
        DB::unprepared(sprintf("
            CREATE TRIGGER before_%s_insert
            BEFORE INSERT
            ON %s FOR EACH ROW
            BEGIN
                IF NEW.temporal_start IS NULL THEN
                    SET NEW.temporal_start = NOW(6);
                END IF;
            END;", $originalTableName, $originalTableName));
    }

    private function createTriggerAfterInsert(string $originalTableName, string $historyTableName, array $columns)
    {
        DB::unprepared(sprintf("
            CREATE TRIGGER after_%s_insert
            AFTER INSERT
            ON %s FOR EACH ROW
            BEGIN
                INSERT INTO %s (%s)
                VALUES (%s);
            END;
        ", $originalTableName, $originalTableName, $historyTableName,
            implode(', ', array_map(function($col){ return "`$col`"; }, $columns)),
            implode(', ', array_map(function($col){ return 'NEW.' . $col; }, $columns))
        ));
    }

    private function createTriggerBeforeUpdate(string $originalTableName, string $historyTableName, array $columns, array $triggerColumns)
    {
        $columns = array_filter($columns, function($col){
            return ! in_array($col, [
                'temporal_start',
                'temporal_end',
            ]);
        });

        $updateColumns = implode(', ', array_map(function($col){ return "`$col`" . '=' . 'NEW.' . $col; }, $columns));
        $updateColumns .= ", temporal_start = NEW.temporal_start";

        $triggerCondition = '(1 = 1)';

        if ( count($triggerColumns) ){
            $triggerCondition = '(';

            if ( Arr::isAssoc($triggerColumns) ){

                $i = 0;
                foreach( $triggerColumns as $column => $value ){
                    if ( $i > 0 ) $triggerCondition .= " OR ";

                    $triggerCondition .= "(NEW.$column <> OLD.$column AND ";
                    if ( is_array($value) ){
                        $values = array_map(function($val){return "'$val'";}, $value);
                        $triggerCondition .= "NEW.$column IN (".implode(',', $values).")";
                    } else {
                        $triggerCondition .= "NEW.$column = '$value'";
                    }
                    $triggerCondition .= ')';
                    $i++;
                }

            } else {
                foreach( $triggerColumns as $i => $column ){
                    if ( $i > 0 ) $triggerCondition .= " OR ";
                    $triggerCondition .= "NEW.$column <> OLD.$column";
                }
            }

            $triggerCondition .= ')';
        }



        DB::unprepared(sprintf("
        CREATE TRIGGER before_%s_update
        BEFORE UPDATE
        ON %s FOR EACH ROW
        BEGIN
            DECLARE now DATETIME(6);
            DECLARE now_history DATETIME(6);

            SET now = NOW(6);
            SET now_history = now - INTERVAL 1 MICROSECOND;

            IF %s THEN
                INSERT INTO %s (%s, temporal_start, temporal_end)
                VALUES (%s, OLD.temporal_start, now_history);
            END IF;

            SET NEW.temporal_start = now;

            UPDATE %s SET
                %s
            WHERE id = OLD.id AND temporal_end IS NULL;
        END;",
            $originalTableName,
            $originalTableName,
            $triggerCondition,
            $historyTableName,
            implode(', ', array_map(function($col){ return "`$col`"; }, $columns)),
            implode(', ', array_map(function($col){ return 'OLD.' . $col; }, $columns)),
            $historyTableName,
            $updateColumns,
        ));
    }

    private function createTriggerAfterDelete(string $originalTableName, string $historyTableName, array $columns)
    {
        DB::unprepared(sprintf("
        CREATE TRIGGER after_%s_delete
        AFTER DELETE
        ON %s FOR EACH ROW
        BEGIN
            UPDATE %s SET
                temporal_end = NOW(6)
            WHERE id = OLD.id AND temporal_end IS NULL;
        END;
        ", $originalTableName, $originalTableName, $historyTableName));
    }

}
