<?php
namespace Bagaar\LaravelTemporalTables\Models\Traits;

use Bagaar\LaravelTemporalTables\Models\Scopes\TemporalScope;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\Concerns\AsPivot;
use Illuminate\Database\Eloquent\Relations\Pivot;

trait HasTemporal
{
    use HasTemporalRelations;

    public static function bootHasTemporal()
    {
        static::addGlobalScope(new TemporalScope);
    }

    public function initializeHasTemporal()
    {
        $this->makeHidden([
            'temporal_start',
            'temporal_end',
            'history_id'
        ]);

        $this->setTable($this->getTable() . '_history');
    }


    public function newModelQuery()
    {
        $query = parent::newModelQuery();

        $query->getQuery()->isTemporalTable = true;

        return $query;
    }


    public static function toTemporalDateTimeString(Carbon $dateTime): string
    {
        return $dateTime->toDateTimeString('microsecond');
    }


    protected function temporalCast(): Attribute
    {
        return Attribute::set(function ($value){
            if (is_a($value, Carbon::class)){
                return self::toTemporalDateTimeString($value);
            }

            return $value;
        });
    }

    protected function temporalStart(): Attribute
    {
        return $this->temporalCast();
    }

    protected function temporalEnd(): Attribute
    {
        return $this->temporalCast();
    }

    /**
     * Reload a fresh model instance from the database.
     *
     * @param  array|string  $with
     * @return static|null
     */
    public function fresh($with = [])
    {
        if (! $this->exists) {
            return;
        }

        return $this->setKeysForSelectQuery($this->newQuery())
            ->useWritePdo()
            ->with(is_string($with) ? func_get_args() : $with)
            ->first();
    }


    /**
     * Reload the current model instance with fresh attributes from the database.
     *
     * @return $this
     */
    public function refresh()
    {
        if (! $this->exists) {
            return $this;
        }

        $this->setRawAttributes(
            $this->setKeysForSelectQuery($this->newQuery())
                ->useWritePdo()
                ->firstOrFail()
                ->attributes
        );

        $this->load(collect($this->relations)->reject(function ($relation) {
            return $relation instanceof Pivot
                || (is_object($relation) && in_array(AsPivot::class, class_uses_recursive($relation), true));
        })->keys()->all());

        $this->syncOriginal();

        return $this;
    }


}
