<?php
namespace Bagaar\LaravelTemporalTables\Models\Traits;

use Bagaar\LaravelTemporalTables\Models\Relations\BelongsTo;
use Bagaar\LaravelTemporalTables\Services\TemporalVersioning\VersioningFacade;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

trait HasTemporalRelations
{

    public static function bootHasTemporalRelations()
    {
        static::saving(
            function ($model) {
                $model->setVersionTimestamp($model);
                $model->syncVersionedModels($model);
            }
        );
    }

    /**
     * Before saving the model set the versioningTimestamp based on a certain condition.
     * Afterwards this will be synced to all the necessary child models.
     * @param $model
     */
    public function setVersionTimestamp($model)
    {
        // only root model should set the timestamp and override this method.
        return;
    }

    /**
     * Iterate over all the relations in the `relations_to_sync_versioning` list
     * and Synchronize the `snapshotted_at` timestamp value to all the child models.
     * This allows ectn specific models to be loaded on there own and still contains all the
     * correctly versioned relations.
     * @param $model
     */
    public function syncVersionedModels($model)
    {
        if ($model->isDirty($this->getVersionTimestampName()) && defined($model::class . '::VERSIONING_SYNC_RELATIONS')) {
            foreach ($model::class::VERSIONING_SYNC_RELATIONS as $relation) {
                // update each model separately, need to trigger saving event for all the models as it works recursively
                foreach ($model->{$relation} as $relatedModel) {
                    $relatedModel->snapshotted_at = $model->snapshotted_at;
                    $relatedModel->save();
                }
            }
        }
    }

    public function isVersioned()
    {
        $versionTimestamp = $this->getAttribute($this->getVersionTimestampName());
        return VersioningFacade::isSnapshotTimestamp($versionTimestamp);
    }

    public function getVersionKeyName()
    {
        return VersioningFacade::getVersionKeyName();
    }

    public function getVersionTimestampName()
    {
        return VersioningFacade::getVersionTimestampName();
    }

    /**
     * This is needed to show the id with timestamp in the jsonApi response
     * @return mixed id OR id-base64(timestamp)
     */
    public function getVersionKey()
    {
        if (!$this->isVersioned()) {
            return $this->getKey();
        }
        return $this->getKey().'-'.base64_encode($this->getAttribute($this->getVersionTimestampName()));
    }


    /**
     * Define an inverse one-to-one or many relationship.
     *
     * @param  string  $related
     * @param  string  $foreignKey
     * @param  string  $ownerKey
     * @param  string  $relation
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function versionedBelongsTo(
        $related,
        $foreignKey = null,
        $ownerKey = null,
        $relation = null
    ) {
        // If no relation name was given, we will use this debug backtrace to extract
        // the calling method's name and use that as the relationship name as most
        // of the time this will be what we desire to use for the relationships.
        if (is_null($relation)) {
            $relation = $this->guessBelongsToRelation();
        }

        $instance = $this->newRelatedInstance($related);

        // If no foreign key was supplied, we can use a backtrace to guess the proper
        // foreign key name by using the name of the relationship function, which
        // when combined with an "_id" should conventionally match the columns.
        if (is_null($foreignKey)) {
            $foreignKey = Str::snake($relation).'_'.$instance->getKeyName();
        }

        // Once we have the foreign key names, we'll just create a new Eloquent query
        // for the related models and returns the relationship instance which will
        // actually be responsible for retrieving and hydrating every relations.
        $ownerKey = $ownerKey ?: $instance->getKeyName();

        return $this->newVersionedBelongsTo(
            $instance->newQuery(),
            $this,
            $foreignKey,
            $ownerKey,
            $relation
        );
    }


    /**
     * Instantiate a new BelongsTo relationship.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  \Illuminate\Database\Eloquent\Model  $child
     * @param  string  $foreignKey
     * @param  string  $ownerKey
     * @param  string  $relation
     * @return BelongsTo
     */
    protected function newVersionedBelongsTo(
        Builder $query,
        Model $child,
        $foreignKey,
        $ownerKey,
        $relation
    ) {
        return new BelongsTo($query, $child, $foreignKey, $ownerKey, $relation);
    }


}
