# laravel-temporal-tables


## Installation

You can install the package via composer:

```bash
composer require bagaar/laravel-temporal-tables
```

The package will automatically register its service provider.

## Usage

### Adding versioning to a table and model
#### Migration
- Include the `TemporalMigrationTrait` trait in the migration
  ```php
  use \Bagaar\LaravelTemporalTables\Database\Migrations\Traits\TemporalMigrationTrait;
  ```
- Add the temporal timestamps to the table
  ```php
    Schema::create('tablename', function (Blueprint $table) {
        //...
        $this->createTemporalDateTimeFields($table);
        //...
    });
  ```
- Create the history table
  ```php
  $this->createTemporalHistoryTable('tablename');
  ```
- don't forget to delete the history table in the down migration
  ```php
  Schema::dropIfExists('tablename_history');
  ```
##### Example of full migration file
```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    use \Bagaar\LaravelTemporalTables\Database\Migrations\Traits\TemporalMigrationTrait;

    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::create('vessels', function (Blueprint $table) {
            $table->id();

            $table->string('name');
            $table->string('imo');

            $this->createTemporalDateTimeFields($table);

            $table->timestamps();
        });

        $this->createTemporalHistoryTable('vessels');
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::dropIfExists('vessels');
        Schema::dropIfExists('vessels_history');
    }
};
```

#### Model
- add the `HasTemporal` trait to the model
```php
<?php
namespace App\Models;

use Bagaar\LaravelTemporalTables\Models\Traits\HasTemporal;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Vessel extends Model
{

    use HasFactory;
    use HasTemporal;

```

### Querying data
When querying a model with temporal data, the results will always be the latest version, same as if versioning wasn't there.

This package adds a `temporal()` function to the Eloquent query builder which can be used to query for a version from a certain timestamp.

```php
Model::find(1); //return the latest version

Model::temporal(now()->subHour())->find(1); //returns version from 1 hour ago
```

#### retrieving all versions
If you want to retrieve all version, pass `true` to the temporal function
```php
Model::temporal(true)->where('id', 1)->get(); //return collection of all versions
```

#### deleted models
If you query a model with a timestamp before a record was deleted, the result will still contain the deleted record
```php
$model = Model::find(1);
$model->delete();

Model::find(1); //will return null

Model::temporal(now()->subMinute())->find(1); //will return a result
```

The same logic applies to newly created models. If you query a model with a timestamp before it was created, it will not exist.
```php
$model = Model::create(['foo' => 'bar']);

Model::find(1); //will return a result

Model::temporal(now()->subMinute())->find(1); //will return null
```

#### retrieving version timestamps
Versioned models have 2 hidden attributes `temporal_start` and `temporal_end`
```php
$model = Model::first();

$model->temporal_start;
$model->temporal_end;
```

### Relationships
Versioned models can be added as a relationship by adding the `HasTemporalRelations` trait and using `versionedBelongsTo()`

When using versioned relationships the "main" model stores a timestamp which is automatically used when retrieving relations.
The default attribute name for the timestamp is `snapshotted_at`, this can be overruled by setting `$version_timestamp_attribute`

```php
protected $version_timestamp_attribute = 'snapshotted_at';
```

Example model
```php
<?php
namespace App\Models;

use Bagaar\LaravelTemporalTables\Models\Traits\HasTemporalRelations;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Ectn extends Model
{
    use HasFactory;
    use HasTemporalRelations;

    public function vessel()
    {
        return $this->versionedBelongsTo(Vessel::class);
    }
}
```


### Joins
When using joins and you are not interested in historical versions, use the `join()` function as it is provided by Eloquent.
```php
Ectn::join('vessels', 'vessels.id', 'ectns.vessel_id')->get()
```

If you want to join a versioned table and use a timestamp constraint, use the `temporalJoin()` function.  
In this case you will have to join the history table instead of the original table.
```php
Ectn::temporalJoin('vessels_history', 'vessels_history.id', '=', 'ectns.vessel_id', 'left', false, $timestamp)->get()
```

You can also join all versions by passing `true` to `temporalJoin()`
```php
Ectn::temporalJoin('vessels_history', 'vessels_history.id', '=', 'ectns.vessel_id', 'left', false, true)->get()
```

### Global versioning
You can use the `VersioningFacade` to set a global timestamp. All models that have versioning will use that timestamp
```php
VersioningFacade::enableGlobalVersioning(now()->subHour());

Model::find(1); //return data from 1 hour ago

VersioningFacade::disableGlobalVersioning();

Model::find(1); //return latest data
```
Global versioning overrules timestamps set using `temporal()`
```php
VersioningFacade::enableGlobalVersioning(now()->subHour());

Model::find(1); //return data from 1 hour ago

//trying to retrieve older data will be overruled by the global timestamp
Model::temporal(now()->subDay())find(1); //return data from 1 hour ago
```


### Manipulating history

#### Manipulating the history table
This package does not provide any functionality to manipulate historical versions, but because all versions are stored in a separate `_history` table nothing stops you from directly manipulating that table.

> **Important, make sure there are no gaps between subsequential records**
> 
> If a records has an `temporal_end` of `2022-09-12 17:10:13.942121`
> 
> The `temporal_start` for the next record should be `2022-09-12 17:10:13.942122`

#### Manipulating temporal_start on insert
If you include the `temporal_start` attribute when creating a model. That timestamp will be used and the database trigger will not generate a new timestamp.

### Conditional versioning
By default, a new version will be created every time a record is updated. 

Versioning can be made conditional by defining which field(s) should trigger a version in the database migration

#### Only version when certain fields are updated
Pass an array of field names to the `createTemporalHistoryTable` function
```php
$this->createTemporalHistoryTable('vessels', ['name']);
```

#### Only version when certain fields are updated with specific values
Pass an array with fields names as keys to the `createTemporalHistoryTable` function

```php
$this->createTemporalHistoryTable('ectns', ['status' => 'approved']);
```

You can also set multiple values
```php
$this->createTemporalHistoryTable('ectns', ['status' => ['approved', 'denied']]);
```

### Using `lockForUpdate()`
This package executes all `SELECT` queries on the `_history` table, when using `lockForUpdate()` the original table should be locked instead of the history. 
This can be solved by manually forcing the query used to retrieve the model.

Instead of
```php
$forwarder = Organisation::whereKey($this->ectn->forwarder_organisation_id)->lockForUpdate()->first();
```
do
```php
$q = \DB::table('organisations')->where('id', $this->ectn->forwarder_organisation_id)->lockForUpdate();
 $forwarder = Organisation::fromQuery($q->toSql(), $q->getBindings())->first();
```


## Test suite

Check https://git.bagaar.be/shelf/back-end/temporal-test for unit tests
