<?php
namespace Bagaar\LaravelTemporalTables\Models\Relations;

use Bagaar\LaravelTemporalTables\Services\TemporalVersioning\VersioningFacade;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;

class BelongsTo extends \Illuminate\Database\Eloquent\Relations\BelongsTo
{

    protected $versioned_models = [];

    public const VERSIONED_KEY_NAME = 'version_key';
    public const VERSIONED_TIMESTAMP = 'snapshotted_at';

    /**
     * Set the base constraints on the relation query.
     *
     * @return void
     */
    public function addConstraints()
    {
        if (static::$constraints) {
            // For belongs to relationships, which are essentially the inverse of has one
            // or has many relationships, we need to actually query on the primary key
            // of the related models matching on the foreign key that's on a parent.
            $table = $this->related->getTable();

            $this->query->where($table . '.' . $this->ownerKey, '=', $this->child->{$this->foreignKey});

            // Check if the child object has a `snapshotted_at` timestamp set.
            $versionTimestamp = $this->child->{self::VERSIONED_TIMESTAMP};
            // This timestamp will be added dynamically to all relation objects to propagate the versioning.
            if ($versionTimestamp) {
                // add temporal query and add versionedKey which can be used when matching.
                $this->query->temporal($versionTimestamp)->select(
                    [
                        '*',
                        DB::raw(
                            'CONCAT(' . $table . '.' . $this->ownerKey . ', "-", TO_BASE64("' . $versionTimestamp . '")) AS ' . self::VERSIONED_KEY_NAME
                        ),
                        DB::raw("'" . $versionTimestamp . "' AS " . self::VERSIONED_TIMESTAMP),
                    ]
                );
            }
        }
    }

    /**
     * Set the constraints for an eager load of the relation.
     *
     * @param array $models
     * @return void
     */
    public function addEagerConstraints(array $models)
    {
        // We'll grab the primary key name of the related models since it could be set to
        // a non-standard name and not "id". We will then construct the constraint for
        // our eagerly loading query so it returns the proper models from execution.
        $key = $this->related->getTable() . '.' . $this->ownerKey;

        //filter versioned and unversioned models
        $unversioned_models = [];
        foreach ($models as $model) {
            // Ectns will load snapshotted_at timestamp from db. For other models a dynamic timestamp is set while retrieving them.
            if ($model->isVersioned()) {
                $this->versioned_models[] = $model;
            } else {
                $unversioned_models[] = $model;
            }
        }

        //clone a new query object before we add the whereIn query
        //this clone will be used in getEager()
        $this->newQuery = clone $this->newQuery();

        $whereIn = $this->whereInMethod($this->related, $this->ownerKey);

        $this->query->{$whereIn}($key, $this->getEagerModelKeys($unversioned_models));

        // Add temporary values to the select to make the union() work in getEager()
        $this->query->select(['*'])->selectRaw(
            $key . ' AS ' . self::VERSIONED_KEY_NAME . ', null AS ' . self::VERSIONED_TIMESTAMP
        );
    }

    /**
     * Get the relationship for eager loading.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getEager()
    {
        $key = $this->related->getTable() . '.' . $this->ownerKey;

        foreach ($this->versioned_models as $model) {
            // get timestamp from request and add models to versioned_models list if request has timestamp
            $timestamp = $model->{self::VERSIONED_TIMESTAMP};

            $q = clone $this->newQuery;

            $q->temporal($timestamp)
                ->where($key, $this->getEagerModelKeys([$model]))
                ->select(['*']);

            if (VersioningFacade::isSnapshotTimestamp($timestamp)) {
                $q->selectRaw(
                    'CONCAT(' . $key . ', "-", TO_BASE64("' . $timestamp . '")) AS ' . self::VERSIONED_KEY_NAME . ', "' . $timestamp . '" AS ' . self::VERSIONED_TIMESTAMP
                );
            } else {
                $q->selectRaw(
                    $key . ' AS ' . self::VERSIONED_KEY_NAME . ', null AS ' . self::VERSIONED_TIMESTAMP
                );
            }

            $this->query->union($q);
        }

        return $this->get();
    }

    /**
     * Match the eagerly loaded results to their parents.
     *
     * @param  array   $models
     * @param  \Illuminate\Database\Eloquent\Collection  $results
     * @param  string  $relation
     * @return array
     */
    public function match(array $models, Collection $results, $relation)
    {
        $foreign = $this->foreignKey;

        // First we will get to build a dictionary of the child models by their primary
        // key of the relationship, then we can easily match the children back onto
        // the parents using that dictionary and the primary key of the children.
        $dictionary = [];

        foreach ($results as $result) {
            $dictionary[$result->getAttribute(self::VERSIONED_KEY_NAME)] = $result;
        }

        // Once we have the dictionary constructed, we can loop through all the parents
        // and match back onto their children using these keys of the dictionary and
        // the primary key of the children to map them onto the correct instances.
        foreach ($models as $model) {
            // Ectn objects will have a `snapshotted_at` attribute stored in the db. This timestamp is used to do the matching
            if ($model->isVersioned()) {
                $versioned_key = $model->{$foreign}.'-'.base64_encode($model->{self::VERSIONED_TIMESTAMP});
            } else {
                $versioned_key = $model->{$foreign};
            }

            if (isset($dictionary[$versioned_key])) {
                $model->setRelation($relation, $dictionary[$versioned_key]);
            } elseif (isset($dictionary[$model->{$foreign}])) {
                $model->setRelation($relation, $dictionary[$model->{$foreign}]);
            }
        }

        return $models;
    }

    protected function getEagerModelKeys(array $models)
    {
        $keys = parent::getEagerModelKeys($models);

        //restored code removed in framework: https://github.com/laravel/framework/commit/fae67da8ad12017ac592c774fb88419d489ba816
        // If there are no keys that were not null we will just return an array with null
        // so this query wont fail plus returns zero results, which should be what the
        // developer expects to happen in this situation. Otherwise we'll sort them.
        if (count($keys) === 0) {
            return [null];
        }

        return $keys;
    }

}
