<?php
namespace Bagaar\LaravelTemporalTables\Services\TemporalVersioning;


use Bagaar\LaravelTemporalTables\Models\Scopes\TemporalScope;
use Bagaar\LaravelTemporalTables\Models\Traits\HasTemporal;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class VersioningService
{
    protected $version_key = 'version_key';
    protected $version_timestamp_attribute = 'snapshotted_at';

    protected ?string $globalVersioningTimestamp;

    /**
     *
     */
    public function __construct()
    {
        $this->globalVersioningTimestamp = null;
    }

    public function getVersionKeyName()
    {
        return $this->version_key;
    }

    public function getVersionTimestampName()
    {
        return $this->version_timestamp_attribute;
    }

    /**
     * @param Carbon|string|bool|null $timestamp
     * @return void
     */
    public function enableGlobalVersioning(Carbon|string|bool|null $timestamp): void
    {
        if ( is_a($timestamp, Carbon::class) ){
            $timestamp = HasTemporal::toTemporalDateTimeString($timestamp);
        }

        $this->globalVersioningTimestamp = $timestamp;
    }

    /**
     * @return void
     */
    public function disableGlobalVersioning(): void
    {
        $this->globalVersioningTimestamp = null;
    }

    /**
     * @return bool
     */
    public function isGlobalVersioningEnabled(): bool
    {
        return !is_null($this->globalVersioningTimestamp);
    }

    /**
     * @return string|bool|null
     */
    public function getGlobalVersioningTimestamp(): ?string
    {
        return $this->globalVersioningTimestamp;
    }

    /**
     * @param $value
     * @return bool
     */
    public function isSnapshotTimestamp(?string $value): bool
    {
        return (!is_null($value) && Carbon::createFromFormat('Y-m-d H:i:s.u', $value) !== false);
    }

    /**
     * @param $builder
     * @param $qualifiedTableName
     * @param $temporalData
     * @return void
     */
    public function addTemporalScopeToQueryBuilder($builder, $qualifiedTableName = null, $temporalData = null)
    {
        if ( ! $temporalData ){
            if ( $this->isGlobalVersioningEnabled() ){
                $temporalData = $this->getGlobalVersioningTimestamp();
            } else {
                $temporalData = $builder->getQuery()->temporal ?? null;
            }
        }

        if ( $temporalData === null){
            $builder->where(function($q) use($qualifiedTableName){
                $q->whereNull($this->qualifyColumn('temporal_end', $qualifiedTableName));
            });
        }

        if ( is_a($temporalData, Carbon::class) ){
            $temporalData = HasTemporal::toTemporalDateTimeString($temporalData);
        }

        if ( is_string($temporalData) ){
            $builder->where(function($q) use($temporalData, $qualifiedTableName){
                $q->whereBetweenColumns(DB::raw("'$temporalData'"), [
                    $this->qualifyColumn('temporal_start', $qualifiedTableName),
                    $this->qualifyColumn('temporal_end', $qualifiedTableName),
                ]);
                $q->orWhere(function($q) use($temporalData, $qualifiedTableName){
                    $q->whereRaw($this->qualifyColumn('temporal_start', $qualifiedTableName) . " <= '" . $temporalData . "'")
                        ->whereNull($this->qualifyColumn('temporal_end', $qualifiedTableName));
                });
            });
        }
    }

    /**
     * @param  string  $column
     * @param  string|null  $table
     * @return string
     */
    protected function qualifyColumn(string $column, ?string $table = null)
    {
        if ( $table ) return $table . '.' . $column;

        return $column;
    }


    /**
     * Fetch applied scopes from model and filter out TemporalScope
     *
     * Meant to be used with withoutGlobalScopes()
     *
     * This allows you to remove all scopes, but still keep the TemporalScope active
     *
     * Ex.: Model::withoutGlobalScopes(\Versioning::keepTemporalScope(Model::class))->get()
     *
     * @param  string  $modelClass
     * @return array
     */
    public function keepTemporalScope(string $modelClass): array
    {
        $scopes = array_keys((new $modelClass)->getGlobalScopes());
        $scopes = array_filter($scopes, function ($e){ return $e != TemporalScope::class; });

        return $scopes;
    }

}
