<?php

namespace Bagaar\Translation;

use Exception;
use GuzzleHttp\Client;
use Illuminate\Support\Collection;
use Symfony\Component\Finder\Finder;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\HttpFoundation\Response;
use GuzzleHttp\Psr7\Response as GuzzleResponse;
use Bagaar\Translation\Exceptions\POEditorException;
use const DIRECTORY_SEPARATOR;

class Translation
{
    protected $apiKey;
    protected $projectId;
    public $referenceLanguage;
    public $referenceLanguageFilename;

    public function __construct()
    {
        $this->referenceLanguage = config('translation.reference_language');
        $this->referenceLanguageFilename = app()->langPath() . DIRECTORY_SEPARATOR . $this->referenceLanguage . '.json';
    }

    public function scan(): Collection
    {
        $allMatches = [];
        $finder = new Finder();

        $finder->in(base_path())
            ->exclude(config('translation.excluded_directories'))
            ->name(config('translation.extensions'))
            ->followLinks()
            ->files();

        /*
         * This pattern is derived from Barryvdh\TranslationManager by Barry vd. Heuvel <barryvdh@gmail.com>
         *
         * https://github.com/barryvdh/laravel-translation-manager/blob/master/src/Manager.php
         */
        $functions = config('translation.functions');
        $pattern =
            // See https://regex101.com/r/jS5fX0/5
            '[^\w]' . // Must not start with any alphanum or _
            '(?<!->)' . // Must not start with ->
            '(' . implode('|', $functions) . ')' . // Must start with one of the functions
            "\(" . // Match opening parentheses
            "\s*" . // Allow whitespace chars after the opening parenthese
            "[\'\"]" . // Match " or '
            '(' . // Start a new group to match:
            '.+' . // Must start with group
            ')' . // Close group
            "[\'\"]" . // Closing quote
            "\s*" . // Allow whitespace chars before the closing parenthese
            "[\),]"  // Close parentheses or new parameter
        ;

        foreach ($finder as $file) {
            if (preg_match_all("/$pattern/siU", $file->getContents(), $matches)) {
                $allMatches[$file->getRelativePathname()] = $matches[2];
            }
        }

        $collapsedKeys = collect($allMatches)->collapse();
        $keys = $collapsedKeys->combine($collapsedKeys);

        $content = $this->getFileContent();
        $keys = $content->union(
            $keys->filter(function ($key) use ($content) {
                return !$content->has($key);
            })
        );

        $newKeys = $keys->diffKeys($content);


        file_put_contents($this->referenceLanguageFilename, json_encode($keys->sortKeys(), JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        return $newKeys;
    }

    public function listProjectLanguages(): Collection
    {
        try {
            $this->setupPoeditorCredentials();
            $response = $this->query('https://api.poeditor.com/v2/languages/list', [
                'form_params' => [
                    'api_token' => $this->apiKey,
                    'id' => $this->projectId
                ]
            ], 'POST');
        } catch (Exception $e) {
            throw $e;
        }
        return collect($response['result']['languages']);
    }

    public function download(array $language): string
    {
        $response = $this->query('https://api.poeditor.com/v2/projects/export', [
            'form_params' => [
                'api_token' => $this->apiKey,
                'id' => $this->projectId,
                'language' => $language['code'],
                'type' => 'key_value_json'
            ]
        ], 'POST');

        $content = collect($this->query($response['result']['url']))
            ->mapWithKeys(function ($entry, $key) {
                return is_array($entry) ? [trim(array_key_first($entry)) => array_pop($entry)] : [$key => trim($entry)];
            })
            ->sortKeys()
            ->toJson(JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

        file_put_contents(app()->langPath() . DIRECTORY_SEPARATOR . $this->getFullLanguageCode($language['code']) . '.json', $content);
        return $language['code'];
    }

    /**
     * Returns a standardised language code format, according to the force_full_language_codes config flag.
     *
     * The given codes from POEditor can range from: it, de, de-ch, en-gb,...
     * and it would be handy if this were more consistent. If the flag is set to true,
     * standardise these to nl-BE, de-DE, it-IT,...
     * The reference language name is never changed.
     *
     * @param  string $languageCode
     * @return string
     */
    public function getFullLanguageCode($languageCode): string
    {
        if (config('translation.force_full_language_codes')) {
            if (!preg_match('/[a-z]{2}-[A-Z]{2}/', $languageCode) && $languageCode != config('translation.reference_language')) {
                if (!str_contains($languageCode, '-')) {
                    return $languageCode . '-' . strtoupper($languageCode);
                } else {
                    $parts = explode('-', $languageCode);
                    return $parts[0] . '-' . strtoupper($parts[1]);
                }
            }
        }
        return $languageCode;
    }

    public function getReferenceTerms(): Collection
    {
        $this->setupPoeditorCredentials();
        return $this->getFileContent()
            ->map(function ($value, $key) {
                return ['term' => $key];
            });
    }

    public function getReferenceTranslations(): Collection
    {
        $this->setupPoeditorCredentials();
        return $this->getFileContent()
            ->map(function ($value, $key) {
                return [
                    'term' => $key,
                    'translation' => [
                        'content' => $value,
                    ],
                ];
            });
    }

    public function addTerms(): array
    {
        try {
            $entries = $this->getReferenceTerms()->toJson();

            return $this->query('https://api.poeditor.com/v2/terms/add', [
                'form_params' => [
                    'api_token' => $this->apiKey,
                    'id' => $this->projectId,
                    'data' => $entries,
                ]
            ], 'POST');
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function addReferenceTranslations(): array
    {
        try {
            $entries = $this->getReferenceTranslations()->toJson();
            return $this->query('https://api.poeditor.com/v2/translations/add', [
                'form_params' => [
                    'api_token' => $this->apiKey,
                    'id' => $this->projectId,
                    'language' => $this->referenceLanguage,
                    'data' => $entries,
                ]
            ], 'POST');
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function updateReferenceTranslations(): array
    {
        try {
            $entries = $this->getReferenceTranslations()->toJson();

            return $this->query('https://api.poeditor.com/v2/translations/update', [
                'form_params' => [
                    'api_token' => $this->apiKey,
                    'id' => $this->projectId,
                    'language' => $this->referenceLanguage,
                    'data' => $entries,
                ]
            ], 'POST');
        } catch (Exception $e) {
            throw $e;
        }
    }

    protected function setupPoeditorCredentials(): void
    {
        if (!$this->apiKey = config('translation.api_key')) {
            throw POEditorException::noApiKey();
        }

        if (!$this->projectId = config('translation.project_id')) {
            throw POEditorException::noProjectId();
        }
    }

    protected function getFileContent(): Collection
    {
        return file_exists($this->referenceLanguageFilename)
            ? collect(json_decode(file_get_contents($this->referenceLanguageFilename), true))
            : collect();
    }

    protected function query($url, $parameters = [], $type = 'GET'): ?array
    {
        try {
            $response = app(Client::class)->request($type, $url, $parameters);
            return $this->handleResponse($response);
        } catch (POEditorException $e) {
            throw POEditorException::communicationError($e->getMessage());
        } catch (GuzzleException $e) {
            throw POEditorException::communicationError($e->getMessage());
        }
    }

    protected function handleResponse(GuzzleResponse $response)
    {
        if (!in_array($response->getStatusCode(), [Response::HTTP_OK, Response::HTTP_CREATED], true)) {
            throw POEditorException::communicationError($response->getBody()->getContents());
        }

        return json_decode($response->getBody()->getContents(), true);
    }
}
