<?php

namespace Bagaar\Translation;

use Bagaar\Translation\Exceptions\POEditorException;
use const DIRECTORY_SEPARATOR;
use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Response as GuzzleResponse;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Symfony\Component\Finder\Finder;
use Symfony\Component\HttpFoundation\Response;

class Translation
{
    protected string $apiKey;

    protected string $projectId;

    public string $referenceLanguage;

    public string $referenceLanguageFilename;

    public function __construct()
    {
        $this->referenceLanguage = config('translation.reference_language');
        $this->referenceLanguageFilename = app()->langPath().DIRECTORY_SEPARATOR.$this->referenceLanguage.'.json';
    }

    public function scan(): Collection
    {
        $allMatches = [];
        $finder = new Finder();

        $finder->in(base_path())
            ->exclude(config('translation.excluded_directories'))
            ->name(config('translation.extensions'))
            ->followLinks()
            ->files();

        /*
         * This pattern is derived from Barryvdh\TranslationManager by Barry vd. Heuvel <barryvdh@gmail.com>
         *
         * https://github.com/barryvdh/laravel-translation-manager/blob/master/src/Manager.php
         */
        $functions = config('translation.functions');
        $pattern =
            // See https://regex101.com/r/jS5fX0/5
            '[^\w]'. // Must not start with any alphanum or _
            '(?<!->)'. // Must not start with ->
            '('.implode('|', $functions).')'. // Must start with one of the functions
            "\(". // Match opening parentheses
            "\s*". // Allow whitespace chars after the opening parenthese
            "[\'\"]". // Match " or '
            '('. // Start a new group to match:
            '.+'. // Must start with group
            ')'. // Close group
            "[\'\"]". // Closing quote
            "\s*". // Allow whitespace chars before the closing parenthese
            "[\),]"  // Close parentheses or new parameter
;

        foreach ($finder as $file) {
            if (preg_match_all("/$pattern/siU", $file->getContents(), $matches)) {
                $allMatches[$file->getRelativePathname()] = $matches[2];
            }
        }

        $collapsedKeys = collect($allMatches)->collapse();
        $keys = $collapsedKeys->combine($collapsedKeys);

        $content = $this->getReferenceFileContent();
        $keys = $content->union(
            $keys->filter(function ($key) use ($content) {
                return ! $content->has($key);
            })
        );

        $newKeys = $keys->diffKeys($content);

        file_put_contents($this->referenceLanguageFilename, json_encode($keys->sortKeys(), JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        return $newKeys;
    }

    public function listProjectLanguages(): Collection
    {
        $this->setupPoeditorCredentials();
        /** @var array $response */
        $response = $this->query('https://api.poeditor.com/v2/languages/list', [
            'form_params' => [
                'api_token' => $this->apiKey,
                'id' => $this->projectId,
            ],
        ], 'POST');

        return collect((array) $response['result']['languages']);
    }

    public function download(array $language): string
    {
        $response = $this->query('https://api.poeditor.com/v2/projects/export', [
            'form_params' => [
                'api_token' => $this->apiKey,
                'id' => $this->projectId,
                'language' => $language['code'],
                'type' => 'key_value_json',
            ],
        ], 'POST');

        $content = collect($this->query($response['result']['url']))
            ->mapWithKeys(function ($entry, $key) {
                return is_array($entry) ? [trim((string) array_key_first($entry)) => array_pop($entry)] : [$key => trim($entry)];
            })
            ->sortKeys()
            ->toJson(JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

        file_put_contents(app()->langPath().DIRECTORY_SEPARATOR.$this->getFullLanguageCode($language['code']).'.json', $content);

        return $language['code'];
    }

    public function getLocalLanguageFiles(): Collection
    {
        $paths = glob(app()->langPath().'/*.json');
        if (! $paths) {
            return Collection::empty();
        }

        return collect($paths)
            ->map(function (string $path, int $_) {
                return [
                    'code' => Str::match('/([A-z]{2,}[-_][A-z]{2,}).json/', $path),
                    'path' => $path,
                ];
            }
            );
    }

    /**
     * Returns a standardised language code format, according to the force_full_language_codes config flag.
     *
     * The given codes from POEditor can range from: it, de, de-ch, en-gb,...
     * and it would be handy if this were more consistent. If the flag is set to true,
     * standardise these to nl-BE, de-DE, it-IT,...
     * The reference language name is never changed.
     *
     * @param  string  $languageCode
     * @return string
     */
    public function getFullLanguageCode($languageCode): string
    {
        if (config('translation.force_full_language_codes')) {
            if (! preg_match('/[a-z]{2}-[A-Z]{2}/', $languageCode) && $languageCode != config('translation.reference_language')) {
                if (! str_contains($languageCode, '-')) {
                    return $languageCode.'-'.strtoupper($languageCode);
                } else {
                    $parts = explode('-', $languageCode);

                    return $parts[0].'-'.strtoupper($parts[1]);
                }
            }
        }

        return $languageCode;
    }

    public function getReferenceTerms(): Collection
    {
        $this->setupPoeditorCredentials();

        return $this->getReferenceFileContent()
            ->map(function ($value, $key) {
                return ['term' => $key];
            });
    }

    public function getReferenceTranslations(): Collection
    {
        $this->setupPoeditorCredentials();

        return $this->getReferenceFileContent()
            ->map(function ($value, $key) {
                return [
                    'term' => $key,
                    'translation' => [
                        'content' => $value,
                    ],
                ];
            });
    }

    public function addTerms(): array
    {
        try {
            $entries = $this->getReferenceTerms()->toJson();

            return $this->query('https://api.poeditor.com/v2/terms/add', [
                'form_params' => [
                    'api_token' => $this->apiKey,
                    'id' => $this->projectId,
                    'data' => $entries,
                ],
            ], 'POST');
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function addReferenceTranslations(): array
    {
        try {
            $entries = $this->getReferenceTranslations()->toJson();

            return $this->query('https://api.poeditor.com/v2/translations/add', [
                'form_params' => [
                    'api_token' => $this->apiKey,
                    'id' => $this->projectId,
                    'language' => $this->referenceLanguage,
                    'data' => $entries,
                ],
            ], 'POST');
        } catch (Exception $e) {
            throw $e;
        }
    }

    public function updateReferenceTranslations(): array
    {
        try {
            $entries = $this->getReferenceTranslations()->toJson();

            return $this->query('https://api.poeditor.com/v2/translations/update', [
                'form_params' => [
                    'api_token' => $this->apiKey,
                    'id' => $this->projectId,
                    'language' => $this->referenceLanguage,
                    'data' => $entries,
                ],
            ], 'POST');
        } catch (Exception $e) {
            throw $e;
        }
    }

    protected function setupPoeditorCredentials(): void
    {
        if (! $this->apiKey = config('translation.api_key')) {
            throw POEditorException::noApiKey();
        }

        if (! $this->projectId = config('translation.project_id')) {
            throw POEditorException::noProjectId();
        }
    }

    protected function getReferenceFileContent(): Collection
    {
        return file_exists($this->referenceLanguageFilename)
            ? collect((array) json_decode((string) file_get_contents($this->referenceLanguageFilename), true))
            : Collection::empty();
    }

    protected function query(string $url, array $parameters = [], string $type = 'GET'): array
    {
        try {
            /** @var \GuzzleHttp\Psr7\Response $response */
            $response = app(Client::class)->request($type, $url, $parameters);

            return $this->handleResponse($response);
        } catch (POEditorException $e) {
            throw POEditorException::communicationError($e->getMessage());
        } catch (GuzzleException $e) {
            throw POEditorException::communicationError($e->getMessage());
        }
    }

    protected function handleResponse(GuzzleResponse $response): array
    {
        if (! in_array($response->getStatusCode(), [Response::HTTP_OK, Response::HTTP_CREATED], true)) {
            throw POEditorException::communicationError($response->getBody()->getContents());
        }

        return json_decode($response->getBody()->getContents(), true);
    }
}
