<?php

namespace Bagaar\Mailchimp\Services;

use DrewM\MailChimp\MailChimp;
use Bagaar\Mailchimp\Events\NewMailchimpSubscription;

class MailchimpService
{
    /**
     * @var MailChimp
     */
    private $mailChimp;

    private $defaultList;
    private $lists = [];

    /**
     * NewsletterService constructor.
     * @param MailChimp $mailChimp
     */
    public function __construct()
    {
        $this->mailChimp = new MailChimp(\Config::get('mailchimp.api_key'));
        $this->defaultList = \Config::get('mailchimp.default_list');
        $this->lists = \Config::get('mailchimp.lists');
    }

    /**
     * @param $email
     * @return string
     */
    private function getSubscriberHash($email)
    {
        return $this->mailChimp->subscriberHash($email);
    }

    /**
     * @param string $list
     * @return mixed
     */
    private function getListId($list = '')
    {
        if (!$list) $list = $this->defaultList;
        return $this->lists[$list]['id'];
    }

    /**
     * @return bool
     */
    public function lastActionSucceeded()
    {
        return !$this->mailChimp->getLastError();
    }

    /**
     * @return array|false
     */
    public function getLastError()
    {
        return $this->mailChimp->getLastError();
    }

    /**
     * @return \DrewM\MailChimp\MailChimp
     */
    public function getApi()
    {
        return $this->mailChimp;
    }

    /**
     * @param $email
     * @param string $list
     * @return array|bool|false
     */
    public function getSubscriber($email, $list = '')
    {
        $list = $this->getListId($list);

        $response = $this->mailChimp->get("lists/{$list}/members/{$this->getSubscriberHash($email)}");

        if ($this->lastActionSucceeded()) return $response;

        return false;
    }


    /**
     * @param $email
     * @param array $mergeFields
     * @param string $list
     * @param array $options
     * @return array|bool|false
     */
    public function subscribeOrUpdate($email, $mergeFields = [], $list = '', $options = [])
    {
        $list = $this->getListId($list);
        $options = $this->getSubscriptionOptions($email, $mergeFields, $options);

        $response = $this->mailChimp->put("lists/{$list}/members/{$this->getSubscriberHash($email)}", $options);

        if (! $this->lastActionSucceeded()) {
            return false;
        }

        return $response;
    }


    /**
     * @param $email
     * @param string $list
     * @return array|bool|false
     */
    public function unsubscribe($email, $list = '')
    {
        $list = $this->getListId($list);

        $response = $this->mailChimp->patch("lists/{$list}/members/{$this->getSubscriberHash($email)}", [
            'status' => 'unsubscribed',
        ]);

        if (! $this->lastActionSucceeded()) {
            return false;
        }
        return $response;
    }

    /**
     * @param $email
     * @param $mergeFields
     * @param $options
     * @return array
     */
    protected function getSubscriptionOptions($email, $mergeFields, $options)
    {
        $defaultOptions = [
            'email_address' => $email,
            'status' => 'subscribed',
            'email_type' => 'html',
        ];
        if (count($mergeFields)) {
            $defaultOptions['merge_fields'] = $mergeFields;
        }
        $options = array_merge($defaultOptions, $options);
        return $options;
    }
}