<?php

namespace Bagaar\Reporting;

use Bagaar\Reporting\Repositories\MeasureRepository;
use Bagaar\Reporting\Repositories\ReportRepository;
use Carbon\Carbon;
use DB;
use Illuminate\Support\Collection;

class Report
{
    protected $config;
    protected $model;
    protected $reportType;
    protected $field;
    protected $startDate;
    protected $endDate;
    protected $identifiers;
    protected $frequency;

    /**
     * @var ReportRepository
     */
    protected $reportRepository;
    /**
     * @var MeasureRepository
     */
    protected $measureRepository;

    /**
     * Report constructor.
     * @param ReportRepository $reportRepository
     * @param MeasureRepository $measureRepository
     */
    public function __construct(ReportRepository $reportRepository, MeasureRepository $measureRepository)
    {
        $this->config = config('bagaar-reporting', []);

        $this->reportRepository = $reportRepository;
        $this->measureRepository = $measureRepository;
    }


    /**
     *
     */
    public function generateAll($modelasked)
    {
        foreach ($this->config as $model) {
            if ( $modelasked && $model['name'] == $modelasked) {
                $this->generateForModel($model);
            } elseif ( ! $modelasked ) {
                $this->generateForModel($model);
            }

        }
    }

    /**
     * @param $model
     */
    public function update($model)
    {
        $modelConfig = array_filter($this->config, function ($configItem) use ($model) {
            return $configItem['class'] === get_class($model);
        });

        $this->generateForModel($modelConfig[0]);
    }


    /**
     * @param $model
     */
    protected function generateForModel($model)
    {
        $this->setModel($model);
        $this->setIdentifiers();

        foreach ($model['frequencies'] as $this->frequency) {
            if (isset($this->model['identifier'])) {
                $this->reportWithIdentifier();
                if ( isset($this->model['fill_gaps']) && $this->model['fill_gaps'] ){
                    $this->fillGapsWithIdentifier();
                }
            } else {
                $this->reportWithoutIdentifier();
                if ( isset($this->model['fill_gaps']) && $this->model['fill_gaps'] ){
                    $this->fillGapsWithoutIdentifier();
                }
            }
        }
    }


    /**
     * @param $model
     */
    protected function setModel($model)
    {
        $this->model = $model;

        if ( ! isset($this->model['timestamp']) ) $this->model['timestamp'] = 'created_at';


        $this->reportRepository->setModelConfig($model);
        $this->measureRepository->setModelConfig($model);
    }


    /**
     * @param array $identifiers
     */
    protected function setIdentifiers(array $identifiers = [])
    {
        if ( ! count( $identifiers ) ){
            $this->identifiers = $this->measureRepository->getIdentifiers();
        } else {
            $this->identifiers = $identifiers;
        }
    }


    /**
     * @param $startStamp
     * @param null $startDate
     * @param null $endDate
     * @return $this
     */
    protected function setDates($startStamp, &$startDate = null, &$endDate = null)
    {
        if ( $startDate === null ){
            $startDate = &$this->startDate;
        }

        if ( $endDate === null ){
            $endDate = &$this->endDate;
        }

        $startDate = Carbon::parse($startStamp);
        switch ($this->frequency) {
            case 'hourly':
                $startDate->startOfDay();
                $endDate = $startDate->copy()->addHour();
                break;
            case 'daily':
                $startDate->startOfDay();
                $endDate = $startDate->copy()->addDay();
                break;
            case 'weekly':
                $startDate->startOfWeek();
                $endDate = $startDate->copy()->addWeek();
                break;
            case 'monthly':
                $startDate->startOfMonth();
                $endDate = $startDate->copy()->addMonth();
                break;
            case 'yearly':
                $startDate->startOfYear();
                $endDate = $startDate->copy()->addYear();
                break;
        }

        return $this;
    }


    /**
     *
     */
    protected function updateDates(&$startDate = null, &$endDate = null)
    {
        if ( $startDate === null ){
            $startDate = &$this->startDate;
        }

        if ( $endDate === null ){
            $endDate = &$this->endDate;
        }

        switch ($this->frequency) {
            case 'hourly':
                $startDate->addHour();
                $endDate = $startDate->copy()->addHour();
                break;
            case 'daily':
                $startDate->addDay();
                $endDate = $startDate->copy()->addDay();
                break;
            case 'weekly':
                $startDate->addWeek();
                $endDate = $startDate->copy()->addWeek();
                break;
            case 'monthly':
                $startDate->addMonth();
                $endDate = $startDate->copy()->addMonth();
                break;
            case 'yearly':
                $startDate->addYear();
                $endDate = $startDate->copy()->addYear();
                break;
        }
    }


    protected function getMeasuresWithExtendedRange($identifier)
    {
        $need_reference = false;
        $startDate = $this->startDate->copy();
        $startDateCopy = $startDate->copy();

        foreach( $this->model['fields'] as $type ){
            if ( $type == 'diff' ){
                $need_reference = true;
                //we always want to compare against the previous day, otherwise there is a partial delta missing
                $startDate->subDay();
                break;
            }
        }

        $results = $this->measureRepository->getMeasuresForRange($identifier, $startDate, $this->endDate);

        if ( $need_reference ){
            //check if the startDate itself is present in results
            //todo this needs to be refactored
            if ( $this->frequency == 'daily' ){
                if( ! count($results) ){
                    return new Collection([]);
                } else {
                    $startDateFound = false;
                    foreach( $results as $result ){
                        if ( $result->{$this->model['timestamp']}->isSameDay($this->startDate) ){
                            $startDateFound = true;
                            break;
                        }
                    }
                    if ( ! $startDateFound ){
                        return new Collection([]);
                    }
                }
            }

            while ( count($results) && count($results) < 2 && $startDate >= $this->measureRepository->getStartTime($identifier) ){
                $startDate->subDay();
                $results = $this->measureRepository->getMeasuresForRange($identifier, $startDate, $this->endDate);
            }

            //first result should be $startDate->subDay() or less
            if ( count($results) ){
                while( $results->first()->{$this->model['timestamp']} >= $startDateCopy && $startDate >= $this->measureRepository->getStartTime($identifier)){
                    $startDate->subDay();
                    $results = $this->measureRepository->getMeasuresForRange($identifier, $startDate, $this->endDate);
                }
            }

            //we only want the last record from previous day
            $count = $results->filter(function($r) use($startDate){
                return (new Carbon($r->{$this->model['timestamp']}))->isSameDay($startDate);
            })->count();

            for($i = 0; $i < $count - 1; $i++){ $results->shift(); }

            if ( count($results) == 1 ){
                $results = new Collection([]);
            }

        }

        return $results;
    }

    protected function fillGapValueFromDivideCount($identifier, Carbon $firstGapDate, Carbon $gapEndDate) {
        $last_report = $this->reportRepository->getReport($identifier, $gapEndDate, $this->frequency);

        $count = 0;
        $tempdate = $firstGapDate->copy();
        $tempdateEnd = $tempdate->copy();

        while( $tempdate <= $gapEndDate ){
            $count++;
            $this->updateDates($tempdate, $tempdateEnd);
        }

        $newValues = [];
        foreach( array_keys($this->model['fields']) as $fieldName ){
            $newValues[$fieldName] = $last_report->{$fieldName} / $count;
        }

        $tempdate = $firstGapDate->copy();
        $tempdateEnd = $tempdate->copy();

        while( $tempdate <= $gapEndDate ){
            $newValues = $this->prepareFilledGapValue($identifier, $tempdate, $this->frequency, $newValues);

            $this->reportRepository->createOrUpdateReport($identifier, $tempdate, $this->frequency, $newValues, true);
            $this->updateDates($tempdate, $tempdateEnd);
        }
    }

    protected function  fillGapValueFromDiffPrev($identifier, Carbon $firstGapDate, Carbon $gapEndDate) {
        $last_report = $this->reportRepository->getReport($identifier, $gapEndDate, $this->frequency);
        $previous_report = $this->reportRepository->getPreviousReport($identifier, $gapEndDate, $this->frequency);

        $count = 0;
        $tempdate = $firstGapDate->copy();
        $tempdateEnd = $tempdate->copy();

        while( $tempdate <= $gapEndDate ){
            $count++;
            $this->updateDates($tempdate, $tempdateEnd);
        }

        $diffvalues = [];
        foreach( array_keys($this->model['fields']) as $fieldName ){
            $diffvalues[$fieldName] = ($last_report->{$fieldName} - $previous_report->{$fieldName}) / $count;
        }


        $prev_values = [];
        foreach( array_keys($this->model['fields']) as $fieldName ){
            $prev_values[$fieldName] = $previous_report->{$fieldName};
        }

        $tempdate = $firstGapDate->copy();
        $tempdateEnd = $tempdate->copy();

        while( $tempdate < $gapEndDate ){
            foreach( array_keys($this->model['fields']) as $fieldName ){
                $prev_values[$fieldName] = $prev_values[$fieldName] + $diffvalues[$fieldName];
            }

            $prev_values = $this->prepareFilledGapValue($identifier, $tempdate, $this->frequency, $prev_values);
            $this->reportRepository->createOrUpdateReport($identifier, $tempdate, $this->frequency, $prev_values, true);
            $this->updateDates($tempdate, $tempdateEnd);
        }
    }

    protected function fillGap($identifier, Carbon $firstGapDate, Carbon $gapEndDate)
    {

        foreach( $this->model['fields'] as $field => $this->reportType ){
            //todo check fillgap for median, high, low
            //todo when different reporttypes in one model differentiate fills
            switch ($this->reportType) {
                case 'average':
                case 'median':
                case 'high':
                case 'low':
                    $this->fillGapValueFromDiffPrev($identifier, $firstGapDate, $gapEndDate);
                    break;
                case 'diff':
                    $this->fillGapValueFromDivideCount($identifier, $firstGapDate, $gapEndDate);
                    break;
            }
            break;
        }
    }


    protected function fillGapWithBetweenValues($identifier, Carbon $firstGapDate, Carbon $gapEndDate)
    {

    }
    
    

    /**
     * Prepares report values that are added between gaps.
     *
     * @param int|null        $identifier
     * @param \Carbon\Carbon  $timestamp  Carbon timestamp
     * @param string          $frequency  daily,weekly,monthly,yearly
     * @param array           $value      Pre-filled value
     *
     * @return array
     */
    protected function prepareFilledGapValue($identifier, $timestamp, $frequency, $value)
    {
        return $value;
    }

    protected function fillGapsWithIdentifier()
    {
        foreach ($this->identifiers as $identifier) {
            $this->setDates($this->measureRepository->getStartTime($identifier));

            $foundgap = false;
            $firstGapDate = null;
            $firstDateCopy = $this->startDate->copy();

            while ($this->startDate < Carbon::now()) {
                $report = $this->reportRepository->getReport($identifier, $this->startDate, $this->frequency);
                if (!$foundgap && !$report && $this->startDate != $firstDateCopy) {
                    $foundgap = true;
                    $firstGapDate = $this->startDate->copy();
                } elseif ($foundgap && $report) {
                    $this->fillGap($identifier, $firstGapDate, $this->startDate->copy());
                    //reset
                    $foundgap = false;
                    $firstGapDate = null;
                }
                $this->updateDates();
            }
        }
    }

    protected function fillGapsWithoutIdentifier()
    {

        $this->setDates($this->measureRepository->getStartTime());

        $foundgap = false;
        $firstGapDate = null;

        while ($this->startDate < Carbon::now()) {
            $report = $this->reportRepository->getReport(null, $this->startDate, $this->frequency);
            if ( !$foundgap && !$report ){
                $foundgap = true;
                $firstGapDate = $this->startDate->copy();
            } else if ( $foundgap && $report ) {
                $this->fillGap(null, $firstGapDate, $this->startDate->copy());
                //reset
                $foundgap = false;
                $firstGapDate = null;
            }

            $this->updateDates();
        }

    }

    /**
     *
     */
    protected function reportWithIdentifier()
    {
        foreach ($this->identifiers as $identifier) {

            $this->setDates($this->measureRepository->getStartTime($identifier));
            $this->reportRepository->removeFilledReports($identifier, $this->frequency);

            while ($this->startDate < Carbon::now()) {
                $measures = $this->getMeasuresWithExtendedRange($identifier);
                $values = $this->calculate($measures);
                if ( $values ){
                    $this->reportRepository->createOrUpdateReport($identifier, $this->startDate, $this->frequency, $values);
                }

                $this->updateDates();
            }
        }
    }


    /**
     *
     */
    protected function reportWithoutIdentifier()
    {
        $this->setDates($this->measureRepository->getStartTime());
        $this->reportRepository->removeFilledReports(null, $this->frequency);

        while ($this->startDate < Carbon::now()) {

            $measures = $this->getMeasuresWithExtendedRange(null);
            $values = $this->calculate($measures);
            if ( $values ){
                $this->reportRepository->createOrUpdateReport(null, $this->startDate, $this->frequency, $values);
            }

            $this->updateDates();
        }
    }


    /**
     * @param $array
     * @return bool|float|mixed
     */
    protected function calculate(Collection $array)
    {
        if (!count($array)) {
            return false;
        }

        $results = [];

        foreach( $this->model['fields'] as $field => $this->reportType ){
            $result = null;

            $values = $array->pluck($field)->toArray();
            switch ($this->reportType) {
                case 'average':
                    $result = Math::average($values);
                    break;
                case 'median':
                    $result = Math::median($values);
                    break;
                case 'high':
                    $result = Math::high($values);
                    break;
                case 'low':
                    $result = Math::low($values);
                    break;
                case 'diff':
                    $result = Math::diff($values);
                    break;
            }

            $results[$field] = $result;
        }

        return $results;
    }

    public function generateForModelWithIdentifier($modelname, $identifiername, $identifiervalue)
    {
        $modeltmp = '';

        foreach ($this->config as $model) {

            if ( strtolower(substr($model['class'],strlen($model['class'])-strlen($modelname),strlen($modelname))) == strtolower($modelname)) {
                $modeltmp = $model;
                break;
            }
        }

        if ( $modeltmp == '' ) {
            return '';
        }

        $this->setModel($modeltmp);
        $this->identifiers[] = $identifiervalue;

        foreach ($model['frequencies'] as $this->frequency) {
            if (isset($this->model['identifier']) && $this->model['identifier'] == $identifiername) {
                $this->reportWithIdentifier();
                if ( isset($this->model['fill_gaps']) && $this->model['fill_gaps'] ){
                    $this->fillGapsWithIdentifier();
                }
            }

        }


    }

}
